/*
 * Copyright oVirt Authors
 * SPDX-License-Identifier: Apache-2.0
*/

package org.ovirt.engine.api.v3.servers;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;

import org.ovirt.engine.api.resource.DataCenterResource;
import org.ovirt.engine.api.restapi.invocation.CurrentManager;
import org.ovirt.engine.api.v3.V3Server;
import org.ovirt.engine.api.v3.types.V3Action;
import org.ovirt.engine.api.v3.types.V3DataCenter;

@Produces({"application/xml", "application/json"})
public class V3DataCenterServer extends V3Server<DataCenterResource> {
    public V3DataCenterServer(DataCenterResource delegate) {
        super(delegate);
    }

    @GET
    public V3DataCenter get() {
        return adaptGet(getDelegate()::get);
    }

    @PUT
    @Consumes({"application/xml", "application/json"})
    public V3DataCenter update(V3DataCenter dataCenter) {
        return adaptUpdate(getDelegate()::update, dataCenter);
    }

    @DELETE
    public Response remove() {
        return adaptRemove(getDelegate()::remove);
    }

    @DELETE
    @Consumes({"application/xml", "application/json"})
    public Response remove(V3Action action) {
        // In version 3 of the API it was possible to remove a data center adding the "force" attribute of the Action
        // included in the request body. But in version 4 of the API body is forbidden for DELETE requests, so this
        // isn't possible. In order to preserve backwards compatibility we need to take the attribute from the body
        // and make it a request parameter.
        if (action != null && action.isSetForce()) {
            CurrentManager.get().getParameters().put("force", action.isForce().toString());
        }
        return adaptRemove(getDelegate()::remove);
    }

    @Path("storagedomains")
    public V3AttachedStorageDomainsServer getStorageDomainsResource() {
        return new V3AttachedStorageDomainsServer(getDelegate().getStorageDomainsResource());
    }

    @Path("clusters")
    public V3ClustersServer getClustersResource() {
        return new V3ClustersServer(getDelegate().getClustersResource());
    }

    @Path("networks")
    public V3DataCenterNetworksServer getNetworksResource() {
        return new V3DataCenterNetworksServer(getDelegate().getNetworksResource());
    }

    @Path("permissions")
    public V3AssignedPermissionsServer getPermissionsResource() {
        return new V3AssignedPermissionsServer(getDelegate().getPermissionsResource());
    }

    @Path("quotas")
    public V3QuotasServer getQuotasResource() {
        return new V3QuotasServer(getDelegate().getQuotasResource());
    }

    @Path("qoss")
    public V3QossServer getQossResource() {
        return new V3QossServer(getDelegate().getQossResource());
    }

    @Path("iscsibonds")
    public V3IscsiBondsServer getIscsiBondsResource() {
        return new V3IscsiBondsServer(getDelegate().getIscsiBondsResource());
    }
}
