/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_select_donau_instances_filtered.c
 * @brief Implementation of the select_donau_instances_filtered function for Postgres
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_select_donau_instances_filtered.h"
#include "taler_merchant_donau.h"
#include "pg_helper.h"

/**
 * Context for select_donau_instances_filtered().
 */
struct SelectDonauInstanceContext
{
  /**
   * Function to call with the results.
   */
  TALER_MERCHANTDB_DonauInstanceFilteredCallback cb;

  /**
   * Closure for @e cb.
   */
  void *cb_cls;

  /**
   * Did database result extraction fail?
   */
  bool extract_failed;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about donau instances.
 *
 * @param[in, out] cls of type `struct SelectDonauInstanceContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
select_donau_instance_cb (void *cls,
                          PGresult *result,
                          unsigned int num_results)
{
  struct SelectDonauInstanceContext *sdc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    char *donau_url;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_string ("donau_url",
                                    &donau_url),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      sdc->extract_failed = true;
      return;
    }
    sdc->cb (sdc->cb_cls,
             donau_url);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_select_donau_instances_filtered (
  void *cls,
  const char *currency,
  TALER_MERCHANTDB_DonauInstanceFilteredCallback cb,
  void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct SelectDonauInstanceContext sdc = {
    .cb = cb,
    .cb_cls = cb_cls,
    /* Can be overwritten by the select_donau_instance_cb */
    .extract_failed = false,
  };

  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (currency),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "select_donau_instances_filtered",
           "SELECT"
           " donau_url"
           " FROM merchant_donau_instances"
           " WHERE (charity_max_per_year).curr = $1");

  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             "select_donau_instances_filtered",
                                             params,
                                             &select_donau_instance_cb,
                                             &sdc);

  /* If there was an error inside select_donau_instance_cb, return a hard error. */
  if (sdc.extract_failed)
    return GNUNET_DB_STATUS_HARD_ERROR;

  return qs;
}
