--
-- This file is part of TALER
-- Copyright (C) 2025 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

-- @file merchant-0025.sql
-- @brief Expand settings to cover new default_wire_transfer_rounding_interval
-- @author Christian Grothoff


BEGIN;

-- Check patch versioning is in place.
SELECT _v.register_patch('merchant-0025', NULL, NULL);

SET search_path TO public;

-- We will need this for LIKE-style pattern matching "gin" indices.
CREATE EXTENSION IF NOT EXISTS pg_trgm;

SET search_path TO merchant, public;


-- See enum GNUNET_TIME_RounderInterval
CREATE TYPE time_rounder_interval
  AS ENUM ('NONE', 'SECOND', 'MINUTE', 'HOUR', 'DAY', 'WEEK', 'MONTH', 'QUARTER', 'YEAR');

ALTER TABLE merchant_instances
  ADD COLUMN default_wire_transfer_rounding_interval time_rounder_interval
    NOT NULL DEFAULT 'NONE'::time_rounder_interval;

COMMENT ON COLUMN merchant_instances.default_wire_transfer_rounding_interval
  IS 'To what round value do we round up wire transfer deadlines computed on the basis of the default_wire_transfer_delay.';

ALTER TABLE merchant_contract_terms
  ALTER COLUMN contract_terms
    TYPE JSONB
    USING convert_from(contract_terms, 'utf-8')::JSONB;

ALTER TABLE merchant_orders
  ALTER COLUMN contract_terms
    TYPE JSONB
    USING convert_from(contract_terms, 'utf-8')::JSONB;

ALTER TABLE merchant_inventory
  ALTER COLUMN description_i18n
    TYPE JSONB
    USING convert_from(description_i18n, 'utf-8')::JSONB,
  ALTER COLUMN taxes
    TYPE JSONB
    USING convert_from(taxes, 'utf-8')::JSONB,
  ALTER COLUMN address
    TYPE JSONB
    USING convert_from(address, 'utf-8')::JSONB;

ALTER TABLE merchant_instances
  ALTER COLUMN jurisdiction
    TYPE JSONB
    USING convert_from(jurisdiction, 'utf-8')::JSONB,
  ALTER COLUMN address
    TYPE JSONB
    USING convert_from(address, 'utf-8')::JSONB;

ALTER TABLE merchant_categories
  ALTER COLUMN category_name_i18n
    TYPE JSONB
    USING convert_from(category_name_i18n, 'utf-8')::JSONB;

ALTER TABLE merchant_token_families
  ALTER COLUMN description_i18n
    TYPE JSONB
    USING convert_from(description_i18n, 'utf-8')::JSONB,
  ALTER COLUMN extra_data
    TYPE JSONB
    USING extra_data::JSONB;

ALTER TABLE merchant_exchange_keys
  ALTER COLUMN keys_json
    TYPE JSONB
    USING keys_json::JSONB;

ALTER TABLE merchant_donau_keys
  ALTER COLUMN keys_json
    TYPE JSONB
    USING keys_json::JSONB;

ALTER TABLE merchant_kyc
  ALTER COLUMN jaccount_limits
    TYPE JSONB
    USING jaccount_limits::JSONB;

ALTER TABLE merchant_template
  ALTER COLUMN editable_defaults
    TYPE JSONB
    USING editable_defaults::JSONB,
  ALTER COLUMN template_contract
    TYPE JSONB
    USING template_contract::JSONB;

ALTER TABLE merchant_exchange_accounts
  ALTER COLUMN debit_restrictions
    TYPE JSONB
    USING debit_restrictions::JSONB,
  ALTER COLUMN credit_restrictions
    TYPE JSONB
    USING credit_restrictions::JSONB;

ALTER TABLE merchant_accounts
  ALTER COLUMN credit_facade_credentials
    TYPE JSONB
    USING credit_facade_credentials::JSONB;

-- The following indices are added to support efficient product filtering.
-- Use a Generalized Inverted Index (GIN) on the respective columns.
-- We use "LOWER" so we can do case-insensitive searches.
-- "gin_trgm_ops" is the operator class for the index, specifically "trigram indexing" from pg_trgm
CREATE INDEX trgm_idx_products_by_name
  ON merchant_inventory
  USING gin (LOWER(product_name) gin_trgm_ops);

CREATE INDEX trgm_idx_products_by_description
  ON merchant_inventory
  USING gin (LOWER(description) gin_trgm_ops);

CREATE INDEX trgm_idx_categories_by_name
  ON merchant_categories
  USING gin (LOWER(category_name) gin_trgm_ops);

CREATE INDEX trgm_idx_contract_summaries
  ON merchant_contract_terms
  USING gin (LOWER((contract_terms ->> 'summary')) gin_trgm_ops);
-- NOTE: Query must use exactly
--   WHERE LOWER(contract_terms ->> 'summary') LIKE LOWER($1);

CREATE INDEX trgm_idx_order_summaries
  ON merchant_orders
  USING gin (LOWER((contract_terms ->> 'summary')) gin_trgm_ops);
-- NOTE: Query must use exactly
--   WHERE LOWER(contract_terms ->> 'summary') LIKE LOWER($1);

-- FIXME: consider
-- --- alas, requires modifications across the plugin...

COMMIT;
