/* 
 *    Programmed By: Mohammed Isam Mohammed [mohammed_isam1984@yahoo.com]
 *    Copyright 2014, 2015, 2016, 2017, 2018 (c)
 * 
 *    file: opensave.c
 *    This file is part of mino.
 *
 *    mino is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    mino is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with mino.  If not, see <http://www.gnu.org/licenses/>.
 */    
#include "defs.h"
#include "kbd.h"
#include "options.h"
#include "edit.h"
#include "modules/modules.h"
#include <pwd.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <fcntl.h>


static int one(const struct dirent *unused) 
{
  return 1;
}

struct passwd *pass;//will be used to find the home dir
int numVisDirs, firstVisDir, selectedDir, totalDirs;
int numVisFiles, firstVisFile, selectedFile, totalFiles;
char **dirs;
char **files;
struct linestruct *lines[MAX_LINES];
FILE *open_file;


#define WRAP_DIR_OUTPUT			                                \
({					                                \
  if(pos >= totalDirs)			                                \
  {					                                \
    fprintf(stdout, "\x1b[%d;%dH%s", (selectedDir >= numVisDirs) ? 	\
    selectedDir-numVisDirs+3:selectedDir+3,				\
    (selectedDir >= numVisDirs) ? x+MAX_DIR_NAME_LEN+1:x+1,		\
    files[pos-totalDirs]);						\
  }									\
  else 					                                \
  {					                                \
    fprintf(stdout, "\x1b[%d;%dH[%s]", (selectedDir >= numVisDirs) ? 	\
    selectedDir-numVisDirs+3:selectedDir+3,				\
    (selectedDir >= numVisDirs) ? x+MAX_DIR_NAME_LEN+1:x+1,		\
    dirs[pos]);								\
  }					                                \
})

#define OUTPUT_NAME(name)                                               \
{                                                                       \
    fprintf(stdout, "\x1b[%d;%dH%s", (selectedDir >= numVisDirs) ?      \
        selectedDir-numVisDirs+3:selectedDir+3,                         \
        (selectedDir >= numVisDirs) ? x+MAX_DIR_NAME_LEN+1:x+1, name);  \
}

void initDirView()
{
    numVisDirs  = SCREEN_H-9;
    firstVisDir = 0;
    selectedDir = -1;
    totalDirs   = 0;
    MAX_DIR_NAME_LEN = (SCREEN_W/2)-8;
}

char *pathAppend(char *head, char *tail)
{
    if(!tail) return NULL;
    int len;
    char *str;
    if(!head)
    {
        if(tail[0] != '/') return NULL;
        goto tail_only;
    }
    else if(tail[0] == '/')
    {
        goto tail_only;
    }
    else
    {
        len = strlen(head)+strlen(tail);
        str = (char *)malloc(len+2);
        if(!str) return NULL;
        strcpy(str, head);
        strcat(str, "/");
        strcat(str, tail);
    }
    return str;
    
tail_only:
    len = strlen(tail);
    str = (char *)malloc(len+1);
    if(!str) return NULL;
    strcpy(str, tail);
    return str;
}

char *createDocTitle(char *open_file_name)
{
    char *name = strrchr(open_file_name, '/');
    if(!name) name = open_file_name;
    else name++;

    if(documentTitle && strcmp(documentTitle, name) == 0) return documentTitle;
    if(documentTitle) free(documentTitle);
    documentTitle = (char *) malloc(strlen(name)+1);
    if(!documentTitle) { msgBox("Insufficient memory", OK, ERROR); return NULL; }
    strcpy(documentTitle, name);
    return documentTitle;
}


/*****************************************
 * openSaveFile():
 * This function shows a dialog box that
 * contains the directory listing of the
 * current dir. It takes over user input
 * to enable navigation through the dir
 * tree and selecting a file to open/save.
 * It then calls _openFile() or _saveFile()
 * to actually open/save the file.
 * ***************************************/
char *openSaveFile(OPEN_SAVE openSave, int showDialog, char *open_file_name)
{
  x = 2; y = 2;
  h = SCREEN_H-1; w = SCREEN_W-1;
  
  char inputName[MAX_DIR_NAME_LEN];//the input file name entered in the field
  int selChar;	//the selected char in input field
  int sel;	//0=dir tree, 1=input field
  char *slash = NULL;
  char *name;
  
  switch(showDialog) 
  {
    case(YES):
        if(dirs ) { free(dirs ); dirs  = NULL; }
        if(files) { free(files); files = NULL; }
        totalDirs = 0; totalFiles = 0;
        //try to open the document directory, if failed open home directory,
        //if also failed, just open the current working directory.
        if(open_file_name) slash = strrchr(open_file_name, '/');
        if(open_file_name && slash)
        {
            char *tmp;
            tmp = (char *) malloc(slash-open_file_name+1);
            if(!tmp) { msgBox("Insufficient memory", OK, ERROR); return NULL; }
            memcpy(tmp, open_file_name, slash-open_file_name);
            tmp[slash-open_file_name] = '\0';
            int res = scanDir(tmp, &dirs, &files, &totalDirs, &totalFiles);
            free(tmp);
            if(!res) return NULL;
        } 
        else if((pass = getpwuid(geteuid()))) 
        {
            if(!scanDir(pass->pw_dir, &dirs, &files, &totalDirs, &totalFiles)) return NULL;
        } 
        else 
        {
            if(!scanDir(".", &dirs, &files, &totalDirs, &totalFiles)) return NULL;
        }
        sel = (openSave == SAVE) ? 1 : 0;
        if(openSave == OPEN) strcpy(inputName, "");
        else 
        {
            //if there is a document title, save the title
            if(documentTitle) strcpy(inputName, documentTitle);
            else              strcpy(inputName, DEFAULT_TITLE);
        }
        selChar = strlen(inputName);
  
        numVisDirs = h-x-3;
        firstVisDir = 0;
        selectedDir = 0;
        refreshDirView();
        //set the input field
        setScreenColorsI(COLOR_WINDOW);
	fprintf(stdout, "\e[%d;%dH", h-2, y+1);
	printf("File: ");
        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
	printf("%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
	fprintf(stdout, "\e[%d;%dH", h-2, (int)(y+strlen(inputName)+7));
	//reset cursor if it is an OPEN file dialog
	if(openSave == OPEN) 
	{
            setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
            int pos = firstVisDir+selectedDir;
            WRAP_DIR_OUTPUT;
	}
        fflush(stdout);
        /***************************************************/
        //take control over user input
        /***************************************************/
        while(1) 
        {
            char *ch = getKey();
            switch(ch[0]) 
            {
                case('a'):
                    if(GNU_DOS_LEVEL > 2 && CTRL) goto do_home;
                    goto do_enter_char; break;
                case(HOME_KEY):
                    if(GNU_DOS_LEVEL > 2) break;
do_home:
                    if(sel == 1) 
                    {
                        selChar = 0;
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                        fflush(stdout);
                    } break;
                case('e'):
                    if(GNU_DOS_LEVEL > 2 && CTRL) goto do_end;
                    goto do_enter_char; break;
                case(END_KEY):
                    if(GNU_DOS_LEVEL > 2) break;
do_end:
                    if(sel == 1) 
                    {
                        selChar = strlen(inputName);
                        if(sel > w-y-9) sel--;
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                        fflush(stdout);
                    } break;
                case('d'):
                    if(GNU_DOS_LEVEL > 3 && CTRL) goto do_del;
                    goto do_enter_char; break;
                case(DEL_KEY):
                    if(GNU_DOS_LEVEL > 3) break;
do_del:
                    if(sel == 1) 
                    {
                        if(selChar == strlen(inputName)) break;
                        int i;
                        for(i = selChar; i < strlen(inputName)-1; i++)
                            inputName[i] = inputName[i+1];
                        inputName[i] = '\0';
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        fprintf(stdout, "\e[%d;%dH", h-2, y+7);
                        fprintf(stdout, "%*s", w-y-9, " ");
                        fprintf(stdout, "\e[%d;%dH", h-2, y+7);
                        fprintf(stdout, "%s", inputName);
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                        fflush(stdout);
                    } break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                case(BACKSPACE_KEY):
                    if(sel == 1) 
                    {
                        if(selChar == 0) break;
                        int i;
                        if(selChar == strlen(inputName)) 
                        {
                            i = strlen(inputName)-1;
                        } 
                        else 
                        {
                            for(i = selChar; i <= strlen(inputName); i++)
                                inputName[i-1] = inputName[i];
                        }
                        inputName[i] = '\0';
                        selChar--;
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        fprintf(stdout, "\e[%d;%dH", h-2, y+7);
                        fprintf(stdout, "%*s", w-y-9, " ");
                        fprintf(stdout, "\e[%d;%dH", h-2, y+7);
                        fprintf(stdout, "%s", inputName);
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                        fflush(stdout);
                    } break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                case(TAB_KEY):
                    sel = (sel == 0) ? 1 : 0;
                    int pos = firstVisDir+selectedDir;
                    if(sel == 1) 
                    {
                        setScreenColorsI(COLOR_WINDOW);
                        WRAP_DIR_OUTPUT;
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                    } 
                    else 
                    {
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        WRAP_DIR_OUTPUT;
                    }
                    break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                case('b'):
                    if(GNU_DOS_LEVEL > 1 && CTRL) goto do_left;
                    goto do_enter_char; break;
                case(LEFT_KEY):
                    if(GNU_DOS_LEVEL > 1) break;
do_left:
                    if(sel == 1) 
                    {
                        if(selChar == 0) break;
                        selChar--;
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                        break;
                    }
	    
                    if(selectedDir >= numVisDirs) 
                    {
                        int pos = firstVisDir+selectedDir;
                        setScreenColorsI(COLOR_WINDOW);
                        WRAP_DIR_OUTPUT;
                        selectedDir -= numVisDirs;
                        pos -= numVisDirs;
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        if(pos >= totalDirs) 
                        {
                            OUTPUT_NAME(files[pos-totalDirs]);
                            strcpy(inputName, files[pos-totalDirs]);
                        } 
                        else 
                        {
                            OUTPUT_NAME(dirs[pos]);
                        }
                    } 
                    else 
                    {
                        if(firstVisDir == 0) break;
                        selectedDir += numVisDirs;
                        firstVisDir -= numVisDirs;
                        refreshDirView();
                    } 
                    setScreenColorsI(COLOR_WINDOW);
                    fprintf(stdout, "\e[%d;%dH", h-2, y+1);
                    fprintf(stdout, "File: ");
                    setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                    fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                    if(sel == 1) fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                    fflush(stdout);
                    break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                case('f'):
                    if(GNU_DOS_LEVEL > 1 && CTRL) goto do_right;
                    goto do_enter_char; break;
                case(RIGHT_KEY):
                    if(GNU_DOS_LEVEL > 1) break;
do_right:
                    if(sel == 1) 
                    {
                        if(selChar == strlen(inputName)) break;
                        if(selChar >= w-y-9) break;
                        selChar++;
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                        break;
                    }
	    
                    if(selectedDir < numVisDirs) 
                    {
                        int pos = firstVisDir+selectedDir;
                        setScreenColorsI(COLOR_WINDOW);
                        WRAP_DIR_OUTPUT;
                        selectedDir += numVisDirs;
                        pos += numVisDirs;
		
                        if(pos >= (totalDirs+totalFiles))
                        {
                            int h = (totalDirs+totalFiles)-pos-1;
                            pos += h;  //h is negative, so add it in order to subtract it!!
                            selectedDir += h;
                        }
		
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        if(pos >= totalDirs) 
                        {
                            OUTPUT_NAME(files[pos-totalDirs]);
                            strcpy(inputName, files[pos-totalDirs]);
                        } 
                        else 
                        {
                            OUTPUT_NAME(dirs[pos]);
                        }
                    } 
                    else 
                    {
                        if((firstVisDir+(numVisDirs*2)) >= (totalDirs+totalFiles)) break;
                        selectedDir -= numVisDirs;
                        firstVisDir += numVisDirs;
                        refreshDirView();
                    } 
                    setScreenColorsI(COLOR_WINDOW);
                    fprintf(stdout, "\e[%d;%dH", h-2, y+1);
                    fprintf(stdout, "File: ");
                    setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                    fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                    if(sel == 1) fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                    fflush(stdout);
                    break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                case('g'):
                    if(GNU_DOS_LEVEL > 2 && CTRL) goto do_esc;
                    goto do_enter_char; break;
                case(ESC_KEY):
                    if(GNU_DOS_LEVEL > 2) break;
do_esc:
                    refreshView();
                    return 0;
                    break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                case(ENTER_KEY):
                    if(sel == 1) 
                    {
                        if(openSave == SAVE) 
                        {
                            struct stat st;
                            int x = lstat(inputName, &st);
                            if(x != -1 && S_ISDIR(st.st_mode))
                            {
                                int i = msgBox("File already exists. Overwrite?", YES|NO, CONFIRM);
                                if(i == NO) 
                                { 
                                    refreshDirView(); 
                                    setScreenColorsI(COLOR_WINDOW);
                                    fprintf(stdout, "\e[%d;%dH", h-2, y+1);
                                    printf("File: ");
                                    setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                                    fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                                    if(sel == 1) fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                                    fflush(stdout);
                                    break; 
                                }
                            }
                            goto saveFileAnGo;
                        }
                        else 
                        {
                            goto openFileAndGo;
                        }
                    }
                    
                    int index = firstVisDir+selectedDir;
                    if(index < totalDirs) 
                    {
                        //selected a directory.. navigate to it
                        scanDir(dirs[index], &dirs, &files, &totalDirs, &totalFiles);
                        firstVisDir = 0;
                        selectedDir = 0;
                        refreshDirView();
                        setScreenColorsI(COLOR_WINDOW);
                        fprintf(stdout, "\e[%d;%dH", h-2, y+1);
                        fprintf(stdout, "File: ");
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        printf("%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                        if(sel == 1) fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                        fflush(stdout);
                    } 
                    else 
                    {
                        //selected a file.. open/save it
                        if(openSave == OPEN) goto openFileAndGo;
                        else 
                        {
                            int i = msgBox("File already exists. Overwrite?", YES|NO, CONFIRM);
                            if(i == NO) 
                            { 
                                refreshDirView(); 
                                setScreenColorsI(COLOR_WINDOW);
                                fprintf(stdout, "\e[%d;%dH", h-2, y+1);
                                printf("File: ");
                                setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                                fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                                if(sel == 1) fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                                fflush(stdout);
                                break; 
                            }
                        }
                        goto saveFileAnGo;
                    }
                    break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                //Navigate up through dirs and files
                case('p'):
                    if(GNU_DOS_LEVEL > 1 && CTRL) goto do_up;
                    goto do_enter_char; break;
                case(UP_KEY):
                    if(GNU_DOS_LEVEL > 1) break;
do_up:
                    if(sel == 1) break;
                    if(selectedDir == 0) 
                    {
                        if(firstVisDir == 0) break;
                        firstVisDir -= numVisDirs;
                        selectedDir = numVisDirs-1;
                        refreshDirView();
                    } 
                    else 
                    {
                        int pos = firstVisDir+selectedDir;
                        setScreenColorsI(COLOR_WINDOW);
                        WRAP_DIR_OUTPUT;
                        selectedDir--; pos--;
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        WRAP_DIR_OUTPUT;
                        if(pos >= totalDirs) 
                        {
                            OUTPUT_NAME(files[pos-totalDirs]);
                            strcpy(inputName, files[pos-totalDirs]);
                        } 
                        else 
                        {
                            OUTPUT_NAME(dirs[pos]);
                        }
                    }
                    fprintf(stdout, "\e[%d;%dH", h-2, y+7);
                    setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                    fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                    if(sel == 1) fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                    fflush(stdout);
                    break;
                /***************************************************/
                /***************************************************/
                /***************************************************/
                //Navigate down through dirs and files
                case('n'):
                    if(GNU_DOS_LEVEL > 1 && CTRL) goto do_down;
                    goto do_enter_char; break;
                case(DOWN_KEY):
                    if(GNU_DOS_LEVEL > 1) break;
do_down:
                    if(sel == 1) break;
                    if(selectedDir == (numVisDirs*2)-1) 
                    {
                        if((firstVisDir+(numVisDirs*2)) < (totalDirs+totalFiles))
                        {
                            firstVisDir += numVisDirs;
                            selectedDir = numVisDirs;
                            refreshDirView();
                            //if SAVE dialog, redraw the input field
                            if(openSave == SAVE) 
                            {
                                setScreenColorsI(COLOR_WINDOW);
                                fprintf(stdout, "\e[%d;%dH", h-2, y+1);
                                fprintf(stdout, "File: ");
                                setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                                fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                                fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                            }
                            fflush(stdout);
                            break;
                        }
                    }
                    else 
                    {
                        int pos = firstVisDir+selectedDir;
                        if(pos >= (totalDirs+totalFiles-1)) break;
                        setScreenColorsI(COLOR_WINDOW);
                        WRAP_DIR_OUTPUT;
                        selectedDir++; pos++;
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        if(pos >= totalDirs) 
                        {
                            OUTPUT_NAME(files[pos-totalDirs]);
                            strcpy(inputName, files[pos-totalDirs]);
                        } 
                        else 
                        {
                            OUTPUT_NAME(dirs[pos]);
                        }
                    }
                    fprintf(stdout, "\e[%d;%dH", h-2, y+7);
                    setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                    fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                    if(sel == 1) fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                    fflush(stdout);
                    break;
                    
                default:
do_enter_char:
                    if(sel == 1) 
                    {
                        if(strlen(inputName) >= w-y-9) break;
                        int i;
                        inputName[strlen(inputName)+1] = '\0';
                        for(i = strlen(inputName); i > selChar; i--)
                            inputName[i] = inputName[i-1];
                        inputName[selChar++] = (CAPS || SHIFT) ? ch[0]-32 : ch[0];
                        fprintf(stdout, "\e[%d;%dH", h-2, y+7);
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        fprintf(stdout, "%s%*s", inputName, (int)(w-y-strlen(inputName)-9), " ");
                        fprintf(stdout, "\e[%d;%dH", h-2, y+selChar+7);
                    }
                    break;
            }//end switch
        }//end while
        break;
    case(NO):
        strcpy(inputName, open_file_name);
        if(!cwd) cwd = getcwd(NULL, 0);
        if(openSave == OPEN) goto openFileAndGo;
        else goto saveFileAnGo;
        break;
  }//end switch
  
saveFileAnGo:
  name = pathAppend(cwd, inputName);
  if(!_saveFile(name))
  {
      free(name);
      return NULL;
  }
  FILE_STATE = SAVED;
  NEW_FILE = 0;
  free(name);
  return open_file_name;
                            
openFileAndGo:
  name = pathAppend(cwd, inputName);
  if(!_openFile(name))
  {
      free(name);
      return NULL;
  }
  FILE_STATE = OPENED;
  NEW_FILE = 0;
  free(name);
  return open_file_name;
}


/***************************************
 * refreshDirView(): 
 * Procedure to refresh the left window
 * showing directory tree.
 * **************************************/
void refreshDirView() 
{
    if(strlen(cwd) > w-y-5) 
    {
        char *tmp;
        tmp = (char *) malloc(w-y+1);
        if(!tmp) { msgBox("Insufficient memory", OK, ERROR); return; }
   
        int i, k, j = w-y-5;
        for(i = strlen(cwd), k = j; k > 1; i--, k--) tmp[k] = cwd[i];
        tmp[0] = '.';
        tmp[1] = '.';
        drawBox(x, y, h, w, tmp, YES, 0);
        free(tmp);
    } else drawBox(x, y, h, w, cwd, YES, 0);
 
    //show control message at the bottom
    fprintf(stdout, "\e[%d;%dH", h-1, y+1);
    setScreenColorsI(COLOR_WINDOW);
    if(GNU_DOS_LEVEL > 2)
        fprintf(stdout, "[ENTER]: Open dir/file [C-p C-n C-f C-b]: Navigate [C-g]: Cancel");
    else if(GNU_DOS_LEVEL > 1)
        fprintf(stdout, "[ENTER]: Open dir/file [C-p C-n C-f C-b]: Navigate [ESC]: Cancel");
    else
        fprintf(stdout, "[ENTER]: Open dir/file  [ARROWS]: Navigate  [ESC]: Cancel");
    fprintf(stdout, "\e[%d;%dH", h-2, y+1);
    fprintf(stdout, "File: ");
 
    int i = firstVisDir;
    int j = 0; int k = 0;
    int curX, curY;
    //int oldSelectedDir = selectedDir;
 
    while(i < totalDirs) 
    {
        if((i-firstVisDir) == selectedDir) 
        {
            setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
            selectedDir = j;
            OUTPUT_NAME(dirs[i]);
            curX = (j >= numVisDirs) ? j-numVisDirs+3 : j+3;
            curY = (j >= numVisDirs) ? x+MAX_DIR_NAME_LEN+strlen(dirs[i]) : x+strlen(dirs[i]);
            curY += 3;
        } 
        else 
        {
            setScreenColorsI(COLOR_WINDOW);
            selectedDir = j;
            OUTPUT_NAME(dirs[i]);
        }
        if(j >= (numVisDirs*2)-1) break;
        i++; j++; k++;
    }
    //if there is more room, show the files
    if((i >= totalDirs) && (j < numVisDirs*2)) 
    {
        if(firstVisDir > totalDirs) j = firstVisDir-totalDirs;
        else j = 0;
    
        while(j < totalFiles) 
        {
            if((i-firstVisDir) == selectedDir) 
            {
                setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                selectedDir = k;
                OUTPUT_NAME(files[j]);
                curX = (j >= numVisDirs) ? j-numVisDirs+3 : j+3;
                curY = (j >= numVisDirs) ? x+MAX_DIR_NAME_LEN+strlen(files[j]) : x+strlen(files[j]);
                curY += 3;
            } 
            else 
            {
                setScreenColorsI(COLOR_WINDOW);
                selectedDir = k;
                OUTPUT_NAME(files[j]);
            }
            if(j >= totalFiles) break;
            if(k >= (numVisDirs*2)-1) break;
            i++; j++; k++;
        }//end while
    }//end if 

    //reposition the cursor
    fprintf(stdout, "\e[%d;%dH", curX, curY);
    fflush(stdout);
}


/***************************************************************
 * This function saves the file specified in open_file_name.
 * It is called indirectly through the function openSaveFile().
 * *************************************************************/
int _saveFile(char *open_file_name) 
{
    if(!(open_file = fopen(open_file_name, "w+"))) 
    {
        return 0;
    }
    
    int i, j;
    for(i = 0; i < totalLines; i++) 
    {
        fprintf(open_file, "%s", lines[i]->text);
    }
    FILE_STATE = SAVED;

    if(!createDocTitle(open_file_name)) return 0;
    fflush(open_file); 

    checkFileExtension(open_file_name);
    return 1;
}

int copyBufToLine(int i, int k, char *buffer, int linked)
{
    lines[i] = allocLineStructB(k);
    if(!lines[i]) return 0;
    lines[i]->linkedToNext = linked;
    memcpy(lines[i]->text, buffer, k);
    lines[i]->text[k] = '\0';
    calcTotalCharsInLine(i);
    return 1;
}

char buf[4096];

/***************************************************************
 * This function opens the file specified in open_file_name.
 * It is called indirectly through the function openSaveFile().
 * *************************************************************/
int _openFile(char *open_file_name) 
{
    if(!(open_file = fopen(open_file_name, "r")))
    {
        return 0;
    }
  
    int i;
    int j = 0;
    
    for(i = 0; i < totalLines; i++)
    {
        free(lines[i]->text);
        free(lines[i]);
    }
    totalLines = 0;

    i = 0;
    int k = 0, l = 0;
    while(!feof(open_file))
    {
        if(totalLines >= MAX_LINES) break;
        j = fgetc(open_file);
        if(j == -1) break;
        if(l >= MAX_CHARS_PER_LINE)
        {
            if(!copyBufToLine(i, k, buf, 1)) return 0;
            i++; k = 0; l = 0;
            totalLines++;
        }
        buf[k++] = j;
        if(j == '\n')
        {
            if(!copyBufToLine(i, k, buf, 0)) return 0;
            i++; k = 0; l = 0;
            totalLines++;
        }
        else if(j == '\t')
        {
            l += TABSPACES(l+1)+1;
        }
        else
        {
            if((j & 0xc0) != 0x80) l++;
        }
    }
    // some remaining chars?
    if(k != 0)
    {
        if(!copyBufToLine(i, k, buf, 0)) return 0;
        totalLines++;
    }

    if(!createDocTitle(open_file_name)) return 0;
    firstVisLine = 0;
    selectedLine = 0;
    selectedChar = 0;
    checkFileExtension(open_file_name);
    fclose(open_file);
    open_file = NULL;
    return 1;
}

void checkFileExtension(char *open_file_name)
{
    AUTO_HIGHLIGHTING = 0;
    //old_window_color = BG_COLOR[COLOR_WINDOW];
    //search for the last dot in the filename
    char *ext = strrchr(open_file_name, '.');
    if(ext != NULL)	//was there a dot?
    {
        //check what was the file extension
        ext++;
	int i, j;
        for(i = 0; i < NUM_MODULES; i++)
        {
            for(j = 0; j < module[i]->extsCount; j++)
            {
                if(strcmp(ext, module[i]->exts[j]) == 0)
                {
                    AUTO_HIGHLIGHTING = 1;
                    BG_COLOR[COLOR_WINDOW] = COLOR_HWINDOW;
                    HIGHLIGHT_MODE = j;
                    curmodule = module[i];
                    break;
                }
            }
            if(AUTO_HIGHLIGHTING) break;
        }
    }
    
    if(!AUTO_HIGHLIGHTING)
    {
        /*
         * No dot in the file name, or file extension not recognized.
         * Check file content.
         */
        char *line = lines[0]->text;
        //is it a shell script file?
        if((strstr(line, "/bin/csh")) || (strstr(line, "/bin/ksh" )) ||
           (strstr(line, "/bin/sh" )) || (strstr(line, "/bin/bash")))
        {
            AUTO_HIGHLIGHTING = 1;
            BG_COLOR[COLOR_WINDOW] = COLOR_HWINDOW;
            HIGHLIGHT_MODE = SHELL_MODE;
            curmodule = module[SHELL_MODE];
        } 
        //is it a PERL script file?
        else if(strstr(line, "/bin/perl"))
        {
            AUTO_HIGHLIGHTING = 1;
            BG_COLOR[COLOR_WINDOW] = COLOR_HWINDOW;
            HIGHLIGHT_MODE = PERL_MODE;
            curmodule = module[PERL_MODE];
        }
        //NONE of the above...
        else 
        { 
            BG_COLOR[COLOR_WINDOW] = old_window_color;
            HIGHLIGHT_MODE = NO_MODE;
            curmodule = &dummyModule;
        }
    }
}
