/*
 * This file is part of Grocy Android.
 *
 * Grocy Android is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Grocy Android is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Grocy Android. If not, see http://www.gnu.org/licenses/.
 *
 * Copyright (c) 2020-2024 by Patrick Zedler and Dominic Zedler
 * Copyright (c) 2024-2026 by Patrick Zedler
 */

package xyz.zedler.patrick.grocy.fragment.bottomSheetDialog;

import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import androidx.annotation.NonNull;
import xyz.zedler.patrick.grocy.R;
import xyz.zedler.patrick.grocy.databinding.FragmentBottomsheetFeedbackBinding;
import xyz.zedler.patrick.grocy.util.ResUtil;
import xyz.zedler.patrick.grocy.util.UiUtil;
import xyz.zedler.patrick.grocy.util.ViewUtil;

public class FeedbackBottomSheet extends BaseBottomSheetDialogFragment implements OnClickListener {

  private final static String TAG = "FeedbackBottomSheet";

  private FragmentBottomsheetFeedbackBinding binding;

  @Override
  public View onCreateView(
      @NonNull LayoutInflater inflater,
      ViewGroup container,
      Bundle savedInstanceState
  ) {
    binding = FragmentBottomsheetFeedbackBinding.inflate(inflater, container, false);

    binding.linearFeedbackIssue.setBackground(
        ViewUtil.getRippleBgListItemSurface(requireContext())
    );
    binding.linearFeedbackRate.setBackground(
        ViewUtil.getRippleBgListItemSurface(requireContext())
    );
    binding.linearFeedbackShare.setBackground(
        ViewUtil.getRippleBgListItemSurface(requireContext())
    );

    ViewUtil.setOnClickListeners(
        this,
        binding.linearFeedbackIssue,
        binding.linearFeedbackRate,
        binding.linearFeedbackShare
    );

    return binding.getRoot();
  }

  @Override
  public void onDestroy() {
    super.onDestroy();
    binding = null;
  }

  @Override
  public void onClick(View v) {
    int id = v.getId();
    if (getViewUtil().isClickDisabled(id)) {
      return;
    }
    performHapticClick();

    if (id == R.id.linear_feedback_issue) {
      startActivity(
          new Intent(Intent.ACTION_VIEW, Uri.parse(getString(R.string.url_github_issues)))
      );
      dismiss();
    } else if (id == R.id.linear_feedback_rate) {
      ViewUtil.startIcon(binding.imageFeedbackRate);
      Uri uri = Uri.parse(
          "market://details?id=" + requireActivity().getApplicationContext().getPackageName()
      );
      Intent goToMarket = new Intent(Intent.ACTION_VIEW, uri);
      goToMarket.addFlags(Intent.FLAG_ACTIVITY_NO_HISTORY |
          Intent.FLAG_ACTIVITY_NEW_DOCUMENT |
          Intent.FLAG_ACTIVITY_MULTIPLE_TASK |
          Intent.FLAG_ACTIVITY_EXCLUDE_FROM_RECENTS);
      new Handler(Looper.getMainLooper()).postDelayed(() -> {
        try {
          startActivity(goToMarket);
        } catch (ActivityNotFoundException e) {
          startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse(
              "http://play.google.com/store/apps/details?id="
                  + requireActivity().getApplicationContext().getPackageName()
          )));
        }
        dismiss();
      }, 400);
    } else if (id == R.id.linear_feedback_share) {
      ResUtil.share(requireContext(), R.string.msg_share);
      dismiss();
    }
  }

  @Override
  public void applyBottomInset(int bottom) {
    binding.linearFeedbackContainer.setPadding(
        binding.linearFeedbackContainer.getPaddingLeft(),
        binding.linearFeedbackContainer.getPaddingTop(),
        binding.linearFeedbackContainer.getPaddingRight(),
        UiUtil.dpToPx(requireContext(), 8) + bottom
    );
  }

  @NonNull
  @Override
  public String toString() {
    return TAG;
  }
}
