package xyz.malkki.neostumbler.scanner.postprocess

import org.junit.Assert.assertEquals
import org.junit.Assert.assertTrue
import org.junit.Test
import xyz.malkki.neostumbler.core.MacAddress
import xyz.malkki.neostumbler.core.Position
import xyz.malkki.neostumbler.core.emitter.BluetoothBeacon
import xyz.malkki.neostumbler.core.emitter.WifiAccessPoint
import xyz.malkki.neostumbler.core.observation.EmitterObservation
import xyz.malkki.neostumbler.core.observation.PositionObservation
import xyz.malkki.neostumbler.core.report.ReportData

class AutoDetectingMovingWifiBluetoothFiltererTest {
    @Test
    fun `Test moving Wi-Fi and Bluetooth are filtered`() {
        val filterer = AutoDetectingMovingWifiBluetoothFilterer(maxDistanceFromExisting = 500.0)

        val position1 =
            PositionObservation(
                Position(
                    latitude = 40.689100,
                    longitude = -74.044300,
                    source = Position.Source.GPS,
                ),
                timestamp = 0,
            )

        val report1 =
            ReportData(
                position = position1,
                cellTowers = emptyList(),
                wifiAccessPoints =
                    listOf(
                        EmitterObservation(
                            WifiAccessPoint(macAddress = MacAddress("01:01:01:01:01:01")),
                            timestamp = 0,
                        ),
                        EmitterObservation(
                            WifiAccessPoint(macAddress = MacAddress("02:02:02:02:02:02")),
                            timestamp = 0,
                        ),
                    ),
                bluetoothBeacons =
                    listOf(
                        EmitterObservation(
                            BluetoothBeacon(
                                macAddress = MacAddress("04:04:04:04:04:04"),
                                signalStrength = -80,
                            ),
                            timestamp = 0,
                        )
                    ),
            )

        filterer.postProcessReport(report1)

        val afterMoving = position1.position.latLng.destination(700.0, 43.15)

        val position2 =
            PositionObservation(
                position =
                    Position(
                        latitude = afterMoving.latitude,
                        longitude = afterMoving.longitude,
                        source = Position.Source.GPS,
                    ),
                timestamp = 0,
            )

        val report2 = report1.copy(position = position2)

        val filteredReport2 = filterer.postProcessReport(report2)

        assertEquals(0, filteredReport2?.wifiAccessPoints?.size)
        assertEquals(0, filteredReport2?.bluetoothBeacons?.size)

        // Test processing the first report again -> content should get filtered
        val filteredReport1 = filterer.postProcessReport(report1)

        assertEquals(0, filteredReport1?.wifiAccessPoints?.size)
        assertEquals(0, filteredReport1?.bluetoothBeacons?.size)
    }

    @Test
    fun `Test stationary Wi-Fi is not filtered`() {
        val filterer = AutoDetectingMovingWifiBluetoothFilterer(maxDistanceFromExisting = 500.0)

        val position1 =
            PositionObservation(
                position =
                    Position(
                        latitude = 40.689100,
                        longitude = -74.044300,
                        source = Position.Source.GPS,
                    ),
                timestamp = 0,
            )

        val report1 =
            ReportData(
                position = position1,
                cellTowers = emptyList(),
                wifiAccessPoints =
                    listOf(
                        EmitterObservation(
                            WifiAccessPoint(macAddress = MacAddress("01:01:01:01:01:01")),
                            timestamp = 0,
                        ),
                        EmitterObservation(
                            WifiAccessPoint(macAddress = MacAddress("02:02:02:02:02:02")),
                            timestamp = 0,
                        ),
                    ),
                bluetoothBeacons = emptyList(),
            )

        filterer.postProcessReport(report1)

        val afterMoving = position1.position.latLng.destination(700.0, 43.15)

        val position2 =
            PositionObservation(
                position =
                    Position(
                        latitude = afterMoving.latitude,
                        longitude = afterMoving.longitude,
                        source = Position.Source.GPS,
                    ),
                timestamp = 0,
            )

        val report2 =
            ReportData(
                position = position2,
                cellTowers = emptyList(),
                bluetoothBeacons = emptyList(),
                wifiAccessPoints =
                    listOf(
                        EmitterObservation(
                            WifiAccessPoint(macAddress = MacAddress("03:03:03:03:03:03")),
                            timestamp = 0,
                        ),
                        EmitterObservation(
                            WifiAccessPoint(macAddress = MacAddress("04:04:04:04:04:04")),
                            timestamp = 0,
                        ),
                    ),
            )

        val filteredReport = filterer.postProcessReport(report2)

        assertEquals(2, filteredReport?.wifiAccessPoints?.size)
        assertTrue(
            filteredReport?.wifiAccessPoints?.any {
                it.emitter.macAddress.value == "03:03:03:03:03:03"
            } == true
        )
    }

    @Test
    fun `Test moving device can be included in a report in non-deterministic mode`() {
        val filterer =
            AutoDetectingMovingWifiBluetoothFilterer(
                maxDistanceFromExisting = 500.0,
                deterministic = false,
            )

        val position1 =
            PositionObservation(
                Position(
                    latitude = 40.689100,
                    longitude = -74.044300,
                    source = Position.Source.GPS,
                ),
                timestamp = 0,
            )

        val report1 =
            ReportData(
                position = position1,
                cellTowers = emptyList(),
                wifiAccessPoints =
                    listOf(
                        EmitterObservation(
                            WifiAccessPoint(macAddress = MacAddress("01:01:01:01:01:01")),
                            timestamp = 0,
                        )
                    ),
                bluetoothBeacons = emptyList(),
            )

        filterer.postProcessReport(report1)

        val afterMoving = position1.position.latLng.destination(700.0, 43.15)

        val position2 =
            PositionObservation(
                position =
                    Position(
                        latitude = afterMoving.latitude,
                        longitude = afterMoving.longitude,
                        source = Position.Source.GPS,
                    ),
                timestamp = 0,
            )

        val report2 = report1.copy(position = position2)

        var notFiltered = false

        @Suppress("UnusedPrivateProperty")
        for (i in 1..1000) {
            if (filterer.postProcessReport(report2)!!.wifiAccessPoints.isNotEmpty()) {
                notFiltered = true
                break
            }
        }

        assertTrue(notFiltered)
    }
}
