package xyz.lepisma.harp.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.readString
import io.github.vinceglb.filekit.writeString
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.drop
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext
import xyz.lepisma.harp.data.FileUnderDir
import xyz.lepisma.harp.data.Journal
import xyz.lepisma.harp.data.JournalEntry
import xyz.lepisma.harp.data.Profile
import xyz.lepisma.harp.data.formatString
import xyz.lepisma.harp.data.fromOrgDocument
import xyz.lepisma.harp.data.orgLineToString
import xyz.lepisma.harp.data.toastNotify
import xyz.lepisma.orgmode.OrgDocument
import xyz.lepisma.orgmode.OrgSection
import xyz.lepisma.orgmode.lexer.OrgLexer
import xyz.lepisma.orgmode.parse

@OptIn(FlowPreview::class)
class ProfileViewModel : ViewModel() {
    private val _file = MutableStateFlow<FileUnderDir?>(null)
    private val _dataDirPath = MutableStateFlow<PlatformFile?>(null)
    private val _buffer = MutableStateFlow<String?>(null)

    val file = _file.asStateFlow()

    // Path to data directory for attachments and documents. If this is not present, the UI
    // disables certain capabilities and shows appropriate notices.
    val dataDirPath = _dataDirPath.asStateFlow()

    // Buffer maintains the currently opened file in plain string format
    val buffer = _buffer.asStateFlow()

    private val _doc: StateFlow<OrgDocument?> =
        buffer
            .map { text ->
                text?.let {
                    try {
                        parse(OrgLexer(text).tokenize())
                    } catch (_: Exception) {
                        toastNotify("Unable to parse the file as valid org mode file")
                        null
                    }
                }
            }
            .flowOn(Dispatchers.Default)
            .stateIn(viewModelScope, SharingStarted.Eagerly, null)

    // We trigger profile parsing on every change in buffer
    val profile: StateFlow<Profile?> =
        _doc
            .map { d ->
                d?.let {
                    val parsingResult = fromOrgDocument(it)
                    if (parsingResult.isSuccess) {
                        parsingResult.getOrNull()
                    } else {
                        toastNotify("Parsing error: ${parsingResult.exceptionOrNull()?.message}")
                        null
                    }
                }
            }
            .flowOn(Dispatchers.Default)
            .stateIn(viewModelScope, SharingStarted.Eagerly, null)

    private val _selectedJournalIdx = MutableStateFlow<Int>(0)
    val selectedJournalIdx = _selectedJournalIdx.asStateFlow()

    val selectedJournal: StateFlow<Journal?> =
        combine(profile, selectedJournalIdx) { profile, idx ->
            profile?.journals?.getOrNull(idx)
        }.stateIn(
            viewModelScope,
            SharingStarted.Eagerly,
            null
        )

    fun setSelectedJournalIdx(idx: Int): Result<Unit> {
        val journals = profile.value?.journals ?: emptyList()

        return if (idx < 0 || idx >= journals.size) {
            Result.failure(IndexOutOfBoundsException("Index $idx out of bound for journals"))
        } else {
            _selectedJournalIdx.value = idx
            Result.success(Unit)
        }
    }

    fun setSelectedJournalByName(journalName: String): Result<Unit> {
        val journals = profile.value?.journals ?: emptyList()
        val idx = journals.indexOfFirst { journal -> journal.name == journalName }
        return if (idx == -1) {
            Result.failure(Exception("Unable to find journal by the name $journalName"))
        } else {
            setSelectedJournalIdx(idx)
        }
    }

    // On every change, debounce and write to file
    init {
        viewModelScope.launch {
            buffer
                .filterNotNull()
                // HACK: This is not correct when working across multiple profiles
                .drop(1)
                .debounce(500)
                .collect { text ->
                    withContext(Dispatchers.Default) {
                        _file.value?.filePath?.writeString(text)
                    }
                }
        }
    }

    // TODO: Check local file and revert buffer
    suspend fun loadProfile(file: FileUnderDir, dataDirPath: PlatformFile?): Profile =
        withContext(Dispatchers.Default) {
            val text = file.filePath.readString()

            _file.value = file
            _buffer.value = text
            setSelectedJournalIdx(0)
            setDataDirPath(dataDirPath)

            profile.filterNotNull().first()
        }

    fun setDataDirPath(dataDirPath: PlatformFile?) {
        _dataDirPath.value = dataDirPath
    }

    private val _selectedTags = MutableStateFlow<Set<String>>(emptySet())
    val selectedTags = _selectedTags.asStateFlow()

    fun clearTagSelection() {
        _selectedTags.update { emptySet() }
    }

    fun selectTag(tag: String) {
        _selectedTags.update { it + tag }
    }

    fun unselectTag(tag: String) {
        _selectedTags.update { it - tag }
    }

    fun toggleTag(tag: String) {
        _selectedTags.update { current ->
            if (tag in current) current - tag else current + tag
        }
    }

    // To avoid multiple edits
    private val bufferMutex = Mutex()

    /**
     * Take an edit function and modify the buffer
     *
     * NOTE: Only this function should be used for making any changes to the buffer
     */
    private suspend fun atomicBufferEdit(editFn: (text: String, doc: OrgDocument, profile: Profile) -> String): Result<Unit> =
        withContext(Dispatchers.Default) {
            bufferMutex.withLock {
                val text = _buffer.value
                    ?: return@withContext Result.failure(IllegalStateException("Buffer not loaded"))

                val doc = _doc.value
                    ?: return@withContext Result.failure(IllegalStateException("Document not parsed"))

                val profile = profile.value
                    ?: return@withContext Result.failure(IllegalStateException("Profile not parsed"))

                runCatching {
                    _buffer.value = editFn(text, doc, profile)
                }
            }
        }

    private fun deleteSectionFromText(text: String, section: OrgSection): String {
        val beg = section.tokens.first().range.first
        val end = section.tokens.last().range.second

        return text.take(beg) + text.substring(end)
    }

    private fun insertUnderSectionInText(text: String, section: OrgSection, childText: String): String {
        val pos = section.heading.tokens.last().range.second
        return text.take(pos) + "\n" + childText + "\n\n" + text.substring(pos)
    }

    /**
     * Change the title of the section to the provided one
     */
    private fun renameSectionInText(text: String, section: OrgSection, newName: String): String {
        val beg = section.heading.title.tokens.first().range.first
        val end = section.heading.title.tokens.last().range.second

        return text.take(beg) + newName + text.substring(end)
    }

    fun findJournalSectionByName(name: String): Result<OrgSection> {
        val doc = _doc.value
            ?: return Result.failure(IllegalStateException("Document is null"))

        val journalsSection =
            doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }
                ?: return Result.failure(NoSuchElementException("Unable to find Journals heading"))

        for (chunk in journalsSection.body) {
            if (chunk is OrgSection && chunk.heading.level.level == 2) {
                // This is a journal
                val title = orgLineToString(chunk.heading.title).trim()
                if (title == name) {
                    return Result.success(chunk)
                }
            }
        }

        return Result.failure(NoSuchElementException("Unable to find journal $name"))
    }

    suspend fun addJournal(journalName: String): Result<Unit> =
        atomicBufferEdit { text, doc, _ ->
            findJournalSectionByName(journalName).onSuccess {
                error("Journal '$journalName' already exists, pick another name")
            }

            val journalText = Journal(
                name = journalName,
                entries = emptyList()
            ).formatString()

            val journalsSection =
                doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }
                    ?: error("Unable to find Journals heading")

            insertUnderSectionInText(text, journalsSection, journalText)
        }

    suspend fun renameJournal(journalName: String, newName: String): Result<Unit> =
        atomicBufferEdit { text, _, _ ->
            val oldJournal = findJournalSectionByName(journalName).getOrThrow()

            renameSectionInText(text, oldJournal, newName)
        }

    suspend fun deleteJournal(journalName: String): Result<Unit> =
        atomicBufferEdit { text, _, _ ->
            val section = findJournalSectionByName(journalName).getOrThrow()
            deleteSectionFromText(text, section)
        }

    private fun findJournalEntrySectionById(entryId: String): Result<OrgSection> {
        val doc = _doc.value
            ?: return Result.failure(IllegalStateException("Document not loaded"))

        val journalsSection =
            doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }

        if (journalsSection == null) {
            return Result.failure(Exception("Unable to find Journals heading"))
        }

        // We find the entry across all journals. This can be narrowed down by journal but we
        // don't maintain current journal context in view model so we will go with this hack for
        // now
        var entrySection: OrgSection? = null
        for (chunk in journalsSection.body) {
            if (chunk is OrgSection && chunk.heading.level.level == 2) {
                entrySection = chunk.body.find { subchunk ->
                    if (subchunk is OrgSection && subchunk.heading.level.level == 3) {
                        val id = subchunk.heading.properties?.map?.get("ID")?.let {
                            orgLineToString(it).trim()
                        }
                        id != null && id == entryId
                    } else {
                        false
                    }
                } as OrgSection?

                if (entrySection != null) {
                    break
                }
            }
        }

        if (entrySection == null) {
            return Result.failure(Exception("Unable to find section with id $entryId"))
        }

        return Result.success(entrySection)
    }

    /**
     * Move a journal entry from one to another
     */
    suspend fun moveJournalEntry(entry: JournalEntry, targetJournal: String): Result<Unit> =
        atomicBufferEdit { text, doc, _ ->
            doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }
                ?: error("Journals section missing")

            val entrySection = findJournalEntrySectionById(entry.uuid).getOrThrow()

            val afterDelete = deleteSectionFromText(text, entrySection)

            // This reparse usage can be made better
            val newDoc = parse(OrgLexer(afterDelete).tokenize())
                ?: error("Parsing error after deletion")

            val newJournals =
                newDoc.content.find {
                    orgLineToString(it.heading.title).trim() == "Journals"
                } ?: error("Journals section missing after delete")

            val targetJournalSection =
                newJournals.body.find {
                    it is OrgSection &&
                            orgLineToString(it.heading.title).trim() == targetJournal
                } as OrgSection?
                    ?: error("Target journal not found")

            insertUnderSectionInText(
                afterDelete,
                targetJournalSection,
                entry.formatString()
            )
        }

    suspend fun deleteJournalEntry(entry: JournalEntry): Result<Unit> =
        atomicBufferEdit { text, _, _ ->
            val section = findJournalEntrySectionById(entry.uuid).getOrThrow()
            deleteSectionFromText(text, section)
        }

    suspend fun addJournalEntry(journalName: String, entry: JournalEntry): Result<Unit> =
        atomicBufferEdit { text, doc, _ ->
            val journals =
                doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }
                    ?: error("Journals section missing")

            val journal =
                journals.body.find {
                    it is OrgSection &&
                            orgLineToString(it.heading.title).trim() == journalName
                } as OrgSection?
                    ?: error("Journal not found")

            insertUnderSectionInText(text, journal, entry.formatString())
    }

    suspend fun editJournalEntry(journalName: String, entryId: String, newEntry: JournalEntry): Result<Unit> =
        atomicBufferEdit { text, doc, _ ->
            val journalsSection =
                doc.content.find {
                    orgLineToString(it.heading.title).trim() == "Journals"
                } ?: error("Journals section missing")

            val journalSection =
                journalsSection.body.find {
                    it is OrgSection && orgLineToString(it.heading.title).trim() == journalName
                } as OrgSection?
                    ?: error("Journal '$journalName' not found")

            val entrySection = findJournalEntrySectionById(entryId).getOrThrow()

            insertUnderSectionInText(
                deleteSectionFromText(text, entrySection),
                journalSection,
                newEntry.formatString()
            )
        }
}