package xyz.lepisma.harp.screens.journals

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.regular.Edit
import compose.icons.fontawesomeicons.regular.PlusSquare
import compose.icons.fontawesomeicons.regular.StickyNote
import compose.icons.fontawesomeicons.regular.TrashAlt
import compose.icons.fontawesomeicons.solid.EllipsisV
import kotlinx.coroutines.launch
import xyz.lepisma.harp.data.Journal
import xyz.lepisma.harp.data.toastNotify
import xyz.lepisma.harp.screens.PlaceholderBackground
import xyz.lepisma.harp.viewmodel.ProfileViewModel

private enum class JournalOp {
    NEW, EDIT
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun JournalView(
    journals: List<Journal>,
    viewModel: ProfileViewModel,
    modifier: Modifier = Modifier
) {
    val selectedJournalIdx by viewModel.selectedJournalIdx.collectAsState()
    val selectedJournal by viewModel.selectedJournal.collectAsState()

    var journalMenuExpanded by remember { mutableStateOf(false) }
    val selectedTags by viewModel.selectedTags.collectAsState()

    // Variable to hold name of journal under creation or edit
    var journalName: String by remember { mutableStateOf("") }
    var journalOp: JournalOp by remember { mutableStateOf(JournalOp.NEW) }
    var showJournalNameDialog by remember { mutableStateOf(false) }

    var showDeleteConfirmation by remember { mutableStateOf(false) }

    val scope = rememberCoroutineScope()

    Column(
        modifier = modifier.fillMaxSize()
    ) {
        Text(
            "This section displays text entries organized under different journals",
            style = MaterialTheme.typography.bodyMedium,
            modifier = Modifier.padding(vertical = 8.dp)
        )
        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.fillMaxWidth()
        ) {
            JournalSelector(
                journals,
                onChange = { viewModel.setSelectedJournalIdx(it) },
                selectedIdx = selectedJournalIdx,
                modifier = Modifier
                    .weight(1f)
                    .padding(bottom = 14.dp)
            )

            Spacer(Modifier.width(8.dp))

            Box {
                IconButton(
                    onClick = { journalMenuExpanded = true }
                ) {
                    Icon(
                        imageVector = FontAwesomeIcons.Solid.EllipsisV,
                        contentDescription = "Journal action menu",
                        modifier = Modifier.size(24.dp)
                    )
                }
                DropdownMenu(
                    expanded = journalMenuExpanded,
                    onDismissRequest = { journalMenuExpanded = false }
                ) {
                    DropdownMenuItem(
                        text = { Text("Rename Journal") },
                        onClick = {
                            journalOp = JournalOp.EDIT
                            journalName = selectedJournal!!.name
                            showJournalNameDialog = true
                            journalMenuExpanded = false
                        },
                        leadingIcon = {
                            Icon(
                                FontAwesomeIcons.Regular.Edit,
                                "Rename",
                                modifier = Modifier.size(12.dp)
                            )
                        }
                    )
                    DropdownMenuItem(
                        text = { Text("Create New Journal") },
                        onClick = {
                            journalOp = JournalOp.NEW
                            journalName = ""
                            showJournalNameDialog = true
                            journalMenuExpanded = false
                        },
                        leadingIcon = {
                            Icon(
                                FontAwesomeIcons.Regular.PlusSquare,
                                "Create",
                                modifier = Modifier.size(12.dp)
                            )
                        }
                    )
                    DropdownMenuItem(
                        text = { Text("Delete Journal") },
                        onClick = {
                            showDeleteConfirmation = true
                            journalMenuExpanded = false
                        },
                        leadingIcon = {
                            Icon(
                                FontAwesomeIcons.Regular.TrashAlt,
                                "Delete",
                                modifier = Modifier.size(12.dp)
                            )
                        })
                }
            }

            if (showJournalNameDialog) {
                AlertDialog(
                    onDismissRequest = { showJournalNameDialog = false},
                    confirmButton = {
                        TextButton(
                            onClick = {
                                when (journalOp) {
                                    JournalOp.EDIT -> scope.launch {
                                        viewModel.renameJournal(selectedJournal!!.name, journalName)
                                            .onFailure {
                                                toastNotify(
                                                    it.message
                                                        ?: "Unable to rename journal '$journalName'"
                                                )
                                            }
                                    }
                                    // Switch to new journal happens automatically since the journal is
                                    // added just after the heading, and re-parsing takes the first
                                    // journal as the primary one(?).
                                    JournalOp.NEW -> scope.launch {
                                        viewModel.addJournal(journalName).onFailure {
                                            toastNotify(
                                                it.message
                                                    ?: "Unable to add journal '$journalName'"
                                            )
                                        }
                                    }
                                }
                                showJournalNameDialog = false
                            },
                            enabled = journalName.isNotEmpty()
                        ) {
                            Text("Confirm")
                        }
                    },
                    title = {
                        Text( when (journalOp) {
                            JournalOp.EDIT -> "Rename Journal"
                            JournalOp.NEW -> "Create New Journal"
                        })
                    },
                    text = {
                        Column {
                            TextField(
                                value = journalName,
                                label = { Text("Journal name") },
                                onValueChange = { journalName = it }
                            )
                        }
                    }
                )
            }

            if (showDeleteConfirmation) {
                if (journals.size == 1) {
                    toastNotify("Can't delete the only journal")
                    showDeleteConfirmation = false
                } else {
                    AlertDialog(
                        onDismissRequest = { showDeleteConfirmation = false },
                        title = { Text("Delete Journal '${selectedJournal!!.name}'?") },
                        text = {
                            Text("Are your sure you want to delete the journal '${selectedJournal!!.name}'? This action cannot be undone.")
                        },
                        confirmButton = {
                            TextButton(onClick = {
                                scope.launch {
                                    viewModel.deleteJournal(selectedJournal!!.name)
                                        .onFailure { exception ->
                                            toastNotify(
                                                exception.message
                                                    ?: "Unable to delete journal '${selectedJournal!!.name}'"
                                            )
                                        }
                                    showDeleteConfirmation = false
                                }
                            }) {
                                Text("Delete")
                            }
                        },
                        dismissButton = {
                            TextButton(onClick = { showDeleteConfirmation = false }) {
                                Text("Cancel")
                            }
                        }
                    )
                }
            }
        }

        val allEntries = selectedJournal?.entries ?: emptyList()

        val selectedEntries = if (selectedTags.isEmpty()) {
            allEntries
        } else {
            allEntries.filter { entry ->
                entry.tags.any { selectedTags.contains(it) }
            }
        }

        if (allEntries.isEmpty()) {
            PlaceholderBackground("No entries in this Journal", FontAwesomeIcons.Regular.StickyNote)
        } else {
            Text(
                "Total ${selectedEntries.size} of ${allEntries.size} entries displayed",
                style = MaterialTheme.typography.labelMedium,
                modifier = Modifier.padding(vertical = 8.dp)
            )

            LazyColumn(modifier = Modifier.fillMaxSize()) {
                items(selectedEntries) { entry ->
                    JournalEntryView(
                        entry,
                        viewModel = viewModel,
                        modifier = Modifier.padding(vertical = 3.dp)
                    )
                }
            }
        }
    }
}