package xyz.lepisma.harp.screens.journals

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.text.selection.SelectionContainer
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.AssistChip
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.regular.ChartBar
import compose.icons.fontawesomeicons.regular.Edit
import compose.icons.fontawesomeicons.regular.TrashAlt
import compose.icons.fontawesomeicons.solid.Clone
import compose.icons.fontawesomeicons.solid.EllipsisV
import kotlinx.coroutines.launch
import xyz.lepisma.harp.data.JournalEntry
import xyz.lepisma.harp.data.toastNotify
import xyz.lepisma.harp.screens.DTSTampView
import xyz.lepisma.harp.viewmodel.ProfileViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun JournalEntryView(
    entry: JournalEntry,
    viewModel: ProfileViewModel,
    modifier: Modifier = Modifier
) {
    var dropdownExpanded by remember { mutableStateOf(false) }
    var showDeleteConfirmation by remember { mutableStateOf(false) }

    var showJournalMoveDialog by remember { mutableStateOf(false) }
    var moveToJournalName: String? by remember { mutableStateOf(null) }

    val profile by viewModel.profile.collectAsState()
    val selectedJournal by viewModel.selectedJournal.collectAsState()
    val scope = rememberCoroutineScope()

    var showFormSheet by remember { mutableStateOf(false) }
    val sheetState = rememberModalBottomSheetState(skipPartiallyExpanded = true)

    if (showDeleteConfirmation) {
        AlertDialog(
            onDismissRequest = { showDeleteConfirmation = false },
            title = { Text("Delete entry?") },
            text = {
                Text("Are you sure you want to delete this entry? This action cannot be undone.")
            },
            confirmButton = {
                TextButton(onClick = {
                    scope.launch {
                        viewModel.deleteJournalEntry(entry).onFailure { exception ->
                            toastNotify(exception.message ?: "Unable to delete journal")
                        }
                        showDeleteConfirmation = false
                        dropdownExpanded = false
                    }
                }) {
                    Text("Delete")
                }
            },
            dismissButton = {
                TextButton(onClick = { showDeleteConfirmation = false }) {
                    Text("Cancel")
                }
            }
        )
    }

    if (showJournalMoveDialog) {
        val selectedJournalName = selectedJournal!!.name
        val journals = profile?.journals?.filter {
                journal -> journal.name != selectedJournalName
        } ?: emptyList()

        if (journals.isEmpty()) {
            toastNotify("Unable to move since there is only one journal")
            showJournalMoveDialog = false
        } else {
            // Start with first selection
            moveToJournalName = journals[0].name

            AlertDialog(
                onDismissRequest = { showJournalMoveDialog = false },
                title = { Text("Move entry?") },
                text = {
                    Column {
                        Text("Move entry from '${selectedJournalName}' to:")

                        JournalSelector(
                            journals,
                            onChange = { idx ->
                                moveToJournalName = journals[idx].name
                            },
                            label = "Target Journal",
                            modifier = Modifier.padding(top = 10.dp)
                        )
                    }
                },
                confirmButton = {
                    TextButton(
                        onClick = {
                            moveToJournalName?.let { target ->
                                scope.launch {
                                    viewModel.moveJournalEntry(entry, target)
                                        .onFailure {
                                            showJournalMoveDialog = false
                                            dropdownExpanded = false
                                            toastNotify(it.message ?: "Unable to move entry")
                                        }
                                        .onSuccess {
                                            showJournalMoveDialog = false
                                            dropdownExpanded = false
                                            viewModel.setSelectedJournalByName(target).onFailure {
                                                toastNotify(it.message ?: "Unable to switch to journal $target")
                                            }
                                        }
                                }
                            }
                        },
                        enabled = moveToJournalName != null
                    ) {
                        Text("Move")
                    }
                },
                dismissButton = {
                    TextButton(onClick = { showJournalMoveDialog = false }) {
                        Text("Cancel")
                    }
                }
            )
        }
    }

    ElevatedCard(
        modifier = Modifier
            .fillMaxWidth()
            .padding(vertical = 8.dp)
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(10.dp),
            verticalAlignment = Alignment.Top
        ) {
            DTSTampView(entry.datetime)

            Column(
                modifier = Modifier
                    .padding(start = 15.dp)
                    .weight(1f)
            ) {
                SelectionContainer {
                    Text(
                        entry.text.trim(),
                        style = MaterialTheme.typography.bodyLarge
                    )
                }

                if (entry.tags.isNotEmpty() || entry.metricValues.isNotEmpty()) {
                    Spacer(Modifier.height(15.dp))
                    FlowRow(
                        horizontalArrangement = Arrangement.spacedBy(6.dp),
                        verticalArrangement = Arrangement.spacedBy(6.dp)
                    ) {
                        entry.tags.forEach { tag ->
                            AssistChip(
                                onClick = {
                                    viewModel.clearTagSelection()
                                    viewModel.selectTag(tag)
                                },
                                label = { Text("#$tag") },
                                modifier = Modifier.height(24.dp)
                            )
                        }

                        entry.metricValues.forEach { mv ->
                            AssistChip(
                                onClick = {},
                                enabled = false,
                                label = { Text("${mv.id} = ${mv.value}") },
                                leadingIcon = {
                                    Icon(
                                        FontAwesomeIcons.Regular.ChartBar,
                                        null,
                                        Modifier.size(12.dp)
                                    )
                                },
                                modifier = Modifier.height(24.dp)
                            )
                        }
                    }
                }
            }
            Box {
                IconButton(onClick = { dropdownExpanded = true }) {
                    Icon(
                        FontAwesomeIcons.Solid.EllipsisV,
                        contentDescription = "Options",
                        modifier = Modifier.size(14.dp)
                    )
                }
                DropdownMenu(
                    expanded = dropdownExpanded,
                    onDismissRequest = { dropdownExpanded = false }
                ) {
                    DropdownMenuItem(
                        text = { Text("Edit") },
                        onClick = { showFormSheet = true },
                        leadingIcon = {
                            Icon(FontAwesomeIcons.Regular.Edit, null, Modifier.size(12.dp))
                        }
                    )
                    DropdownMenuItem(
                        text = { Text("Move") },
                        onClick = { showJournalMoveDialog = true },
                        leadingIcon = {
                            Icon(FontAwesomeIcons.Solid.Clone, null, Modifier.size(12.dp))
                        }
                    )
                    DropdownMenuItem(
                        text = { Text("Delete") },
                        onClick = { showDeleteConfirmation = true },
                        leadingIcon = {
                            Icon(FontAwesomeIcons.Regular.TrashAlt, null, Modifier.size(12.dp))
                        }
                    )
                }
            }
        }
    }

    if (showFormSheet) {
        ModalBottomSheet(
            onDismissRequest = { showFormSheet = false },
            sheetState = sheetState
        ) {
            if (selectedJournal == null) {
                toastNotify("No journal selected, aborting")
                return@ModalBottomSheet
            }
            val journalName = selectedJournal!!.name

            JournalEntryForm(
                journalName = journalName,
                initialEntry = entry,
                title = "Edit entry",
                onSave = { newEntry ->
                    scope.launch {
                        viewModel.editJournalEntry(
                            journalName = journalName,
                            entryId = entry.uuid,
                            newEntry = newEntry
                        ).onFailure { exception ->
                            toastNotify(
                                exception.message
                                    ?: "Unable to edit journal entry"
                            )
                        }
                        showFormSheet = false
                    }
                }
            )
        }
    }
}
