package xyz.lepisma.harp.screens.journals

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material3.Button
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TimeInput
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.material3.rememberTimePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.regular.Calendar
import kotlinx.coroutines.delay
import kotlinx.datetime.Instant
import kotlinx.datetime.LocalDate
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.TimeZone
import kotlinx.datetime.format
import kotlinx.datetime.format.MonthNames
import kotlinx.datetime.format.char
import kotlinx.datetime.toLocalDateTime
import xyz.lepisma.harp.data.JournalEntry
import kotlin.time.ExperimentalTime
import kotlin.uuid.ExperimentalUuidApi
import kotlin.uuid.Uuid

@OptIn(ExperimentalMaterial3Api::class, ExperimentalUuidApi::class, ExperimentalTime::class)
@Composable
fun JournalEntryForm(
    journalName: String,
    initialEntry: JournalEntry? = null,
    title: String = "Add entry to '$journalName'",
    onSave: (JournalEntry) -> Unit
) {
    val textFocusRequester = remember { FocusRequester() }
    val entryDateTime = initialEntry?.datetime ?: kotlin.time.Clock.System.now().toLocalDateTime(TimeZone.currentSystemDefault())
    var entryDate by remember { mutableStateOf(entryDateTime.date) }
    val dateState = rememberDatePickerState()
    val timeState = rememberTimePickerState(
        initialHour = entryDateTime.time.hour,
        initialMinute = entryDateTime.time.minute,
        is24Hour = true
    )

    var text by remember { mutableStateOf(initialEntry?.text ?: "") }

    LaunchedEffect(Unit) {
        delay(50)
        textFocusRequester.requestFocus()
    }

    var showDatePicker by remember { mutableStateOf(false) }

    Column(
        modifier = Modifier
            .fillMaxWidth()
            .padding(15.dp),
        verticalArrangement = Arrangement.spacedBy(10.dp)
    ) {
        Text(
            text = title,
            style = MaterialTheme.typography.headlineSmall,
            color = MaterialTheme.colorScheme.outline,
            modifier = Modifier.padding(bottom = 20.dp)
        )

        OutlinedTextField(
            value = text,
            onValueChange = { text = it },
            placeholder = { Text("Use #hashtags for tagging and #metric-id(number) to log metric values.") } ,
            label = { Text("Text") },
            modifier = Modifier
                .fillMaxWidth()
                .height(200.dp)
                .focusRequester(textFocusRequester)
        )

        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.spacedBy(12.dp)
        ) {

            TimeInput(
                state = timeState,
                modifier = Modifier.weight(1f)
            )

            OutlinedTextField(
                value = entryDate.format(LocalDate.Format {
                    monthName(MonthNames.ENGLISH_ABBREVIATED)
                    char(' ')
                    dayOfMonth()
                    chars(", ")
                    year()
                }),
                readOnly = true,
                onValueChange = { },
                label = { Text("Date") },
                modifier = Modifier
                    .weight(1f),
                trailingIcon = {
                    IconButton(onClick = { showDatePicker = true }) {
                        Icon(
                            FontAwesomeIcons.Regular.Calendar,
                            null,
                            modifier = Modifier.size(14.dp)
                        )
                    }
                }
            )

            if (showDatePicker) {
                DatePickerDialog(
                    onDismissRequest = { showDatePicker = false },
                    confirmButton = {
                        TextButton(onClick = {
                            val sel = dateState.selectedDateMillis
                            if (sel != null) {
                                entryDate = Instant.fromEpochMilliseconds(sel)
                                    .toLocalDateTime(TimeZone.currentSystemDefault()).date
                            }
                            showDatePicker = false
                        }) { Text("Okay") }
                    }
                ) {
                    DatePicker(dateState)
                }
            }
        }

        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.spacedBy(12.dp)
        ) {
            Spacer(Modifier.weight(1f))
            Button(
                onClick = {
                    onSave(JournalEntry(
                        datetime = LocalDateTime(
                            year = entryDate.year,
                            monthNumber = entryDate.monthNumber,
                            dayOfMonth = entryDate.dayOfMonth,
                            hour = timeState.hour,
                            minute = timeState.minute,
                            second = 0
                        ),
                        uuid = Uuid.random().toHexDashString(),
                        tags = emptyList(),
                        metricValues = emptyList(),
                        text = text,
                        assets = emptyList(),
                        isPrivate = false
                    ))
                },
                modifier = Modifier.weight(1f)
            ) { Text("Save") }
        }
    }
}