package xyz.lepisma.harp.screens.home

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.safeContentPadding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import cafe.adriel.voyager.core.screen.Screen
import cafe.adriel.voyager.navigator.LocalNavigator
import cafe.adriel.voyager.navigator.currentOrThrow
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.regular.FolderOpen
import compose.icons.fontawesomeicons.regular.PlusSquare
import compose.icons.fontawesomeicons.regular.TimesCircle
import compose.icons.fontawesomeicons.solid.Cat
import compose.icons.fontawesomeicons.solid.Cross
import compose.icons.fontawesomeicons.solid.Dog
import compose.icons.fontawesomeicons.solid.Eraser
import compose.icons.fontawesomeicons.solid.Exclamation
import compose.icons.fontawesomeicons.solid.User
import kotlinx.coroutines.launch
import xyz.lepisma.harp.LocalRecentProfiles
import xyz.lepisma.harp.data.ProfileInfo
import xyz.lepisma.harp.data.ProfileType
import xyz.lepisma.harp.screens.PlaceholderBackground
import xyz.lepisma.harp.screens.ProfileScreen

object HomeScreen : Screen {
    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    override fun Content() {
        val navigator = LocalNavigator.currentOrThrow
        val recentProfiles = LocalRecentProfiles.current

        val scope = rememberCoroutineScope()

        /**
         * Open a profile using its uuid, assuming the data is already remembered in the cache
         */
        fun openProfile(profileUuid: String) {
            navigator.push(ProfileScreen(profileUuid))
        }

        /**
         * Load a profile info structure and navigate to it
         */
        suspend fun rememberAndOpenProfile(profileInfo: ProfileInfo) {
            // First we remember this profileInfo and then ask the profile screen to load the UI
            recentProfiles.rememberProfile(profileInfo)
            openProfile(profileInfo.uuid)
        }

        Surface(
            modifier = Modifier.fillMaxSize(),
            color = MaterialTheme.colorScheme.background,
            contentColor = MaterialTheme.colorScheme.onBackground
        ) {
            Column(
                modifier = Modifier
                    .safeContentPadding()
                    .fillMaxSize(),
            ) {
                HarpTitle()

                Spacer(Modifier.height(10.dp))

                Text(
                    text = "Recent profiles",
                    style = MaterialTheme.typography.titleMedium
                )

                Spacer(Modifier.height(8.dp))

                if (recentProfiles.all().isEmpty()) {
                    PlaceholderBackground(
                        "No recently opened profiles",
                        FontAwesomeIcons.Solid.Exclamation,
                        modifier = Modifier.weight(1f)
                    )
                } else {
                    var forgetDialogOpen by remember { mutableStateOf(false) }
                    var profileInfoToForget by remember { mutableStateOf<ProfileInfo?>(null) }

                    recentProfiles.all().forEach { profileInfo ->
                        ElevatedCard(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(vertical = 8.dp),
                            shape = MaterialTheme.shapes.small
                        ) {
                            Row(
                                verticalAlignment = Alignment.CenterVertically,
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .clickable {
                                        openProfile(profileInfo.uuid)
                                    }
                            ) {
                                Icon(
                                    when (profileInfo.type) {
                                        ProfileType.CAT -> FontAwesomeIcons.Solid.Cat
                                        ProfileType.DOG -> FontAwesomeIcons.Solid.Dog
                                        else -> FontAwesomeIcons.Solid.User
                                    },
                                    contentDescription = "Profile Icon",
                                    modifier = Modifier
                                        .padding(start = 12.dp)
                                        .size(12.dp),
                                    tint = MaterialTheme.colorScheme.outline
                                )
                                Text(
                                    profileInfo.name,
                                    fontWeight = FontWeight.Companion.SemiBold,
                                    style = MaterialTheme.typography.titleLarge,
                                    modifier = Modifier.padding(12.dp)
                                )
                                Spacer(Modifier.weight(1f))
                                IconButton({
                                    profileInfoToForget = profileInfo
                                    forgetDialogOpen = true
                                }) {
                                    Icon(
                                        imageVector = FontAwesomeIcons.Solid.Eraser,
                                        contentDescription = "Forget profile button",
                                        modifier = Modifier.size(12.dp),
                                        tint = MaterialTheme.colorScheme.error
                                    )
                                }
                            }
                        }
                    }

                    if (forgetDialogOpen && profileInfoToForget != null) {
                        AlertDialog(
                            onDismissRequest = { forgetDialogOpen = false },
                            confirmButton = {
                                TextButton(onClick = {
                                    recentProfiles.forgetProfile(profileInfoToForget!!.uuid)
                                    profileInfoToForget = null
                                    forgetDialogOpen = false
                                }) {
                                    Text("Forget")
                                }
                            },
                            dismissButton = {
                                TextButton(onClick = { forgetDialogOpen = false }) {
                                    Text("Dismiss")
                                }
                            },
                            title = { Text("Forget profile ${profileInfoToForget!!.name}?") },
                            text = {
                                Column {
                                    Text("Forgetting a profile doesn't delete any data from your device. You can always open the profile by relocating the main file.")
                                }
                            }
                        )
                    }
                }

                Spacer(Modifier.weight(1f))

                Row(
                    Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.spacedBy(12.dp)
                ) {
                    var showOpenProfileDialog by remember { mutableStateOf(false) }
                    if (showOpenProfileDialog) {
                        OpenProfileDialog(
                            onCompletion = { profileInfo ->
                                scope.launch { rememberAndOpenProfile(profileInfo) }
                            },
                            onDismiss = { showOpenProfileDialog = false }
                        )
                    }

                    Button(
                        onClick = { showOpenProfileDialog = true },
                        modifier = Modifier.weight(1f)
                    ) {
                        Icon(
                            FontAwesomeIcons.Regular.FolderOpen,
                            contentDescription = "Open",
                            modifier = Modifier.size(20.dp)
                        )
                        Spacer(Modifier.width(6.dp))
                        Text("Open")
                    }

                    var showNewProfileDialog by remember { mutableStateOf(false) }
                    if (showNewProfileDialog) {
                        NewProfileForm(
                            onCompletion = { profileInfo ->
                                showNewProfileDialog = false
                                scope.launch { rememberAndOpenProfile(profileInfo) }
                            },
                            onDismiss = { showNewProfileDialog = false }
                        )
                    }

                    Button(
                        onClick = { showNewProfileDialog = true },
                        modifier = Modifier.weight(1f)
                    ) {
                        Icon(
                            FontAwesomeIcons.Regular.PlusSquare,
                            contentDescription = "New",
                            modifier = Modifier.size(20.dp)
                        )
                        Spacer(Modifier.width(6.dp))
                        Text("Create")
                    }
                }
            }
        }
    }
}
