package xyz.lepisma.harp.screens.documents

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.text.selection.SelectionContainer
import androidx.compose.material3.AssistChip
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.regular.ChartBar
import compose.icons.fontawesomeicons.solid.EllipsisV
import xyz.lepisma.harp.data.Document
import xyz.lepisma.harp.screens.AssetChip
import xyz.lepisma.harp.screens.DTSTampView
import xyz.lepisma.harp.screens.DTStampInlineView
import xyz.lepisma.harp.viewmodel.ProfileViewModel

@Composable
fun DocumentCard(
    document: Document,
    viewModel: ProfileViewModel,
    modifier: Modifier = Modifier
) {
    var dropdownExpanded by remember { mutableStateOf(false) }
    val tags = document.tags.filter { text -> text != "ATTACH" }

    ElevatedCard(
        modifier = Modifier.Companion
            .fillMaxWidth()
            .padding(vertical = 8.dp)
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(10.dp),
            verticalAlignment = Alignment.Top
        ) {
            DTSTampView(document.datetime)

            Column(
                modifier = Modifier
                    .padding(start = 15.dp)
                    .weight(1f)
            ) {
                Text(
                    text = document.name,
                    style = MaterialTheme.typography.titleLarge
                )
                if (document.source != null) {
                    Text(
                        text = "from ${document.source.id}",
                        fontStyle = FontStyle.Companion.Italic,
                        modifier = Modifier.Companion.padding(top = 5.dp)
                    )
                }

                Spacer(Modifier.Companion.height(10.dp))

                // Main file list
                document.assets.forEachIndexed { index, asset ->
                    Row(verticalAlignment = Alignment.Companion.CenterVertically) {
                        Text(
                            "${index + 1}.",
                            modifier = Modifier.Companion.padding(end = 6.dp)
                        )
                        AssetChip(asset, document.uuid, viewModel)
                    }
                }

                if (document.annotation != null) {
                    SelectionContainer {
                        Text(
                            document.annotation.trim(),
                            style = MaterialTheme.typography.bodyLarge,
                            modifier = Modifier.Companion.padding(15.dp)
                        )
                    }
                }

                if (tags.isNotEmpty() || document.metricValues.isNotEmpty()) {
                    Spacer(Modifier.height(15.dp))
                    FlowRow(
                        horizontalArrangement = Arrangement.spacedBy(6.dp),
                        verticalArrangement = Arrangement.spacedBy(6.dp)
                    ) {
                        tags.forEach { tag ->
                            AssistChip(
                                onClick = {
                                    viewModel.clearTagSelection()
                                    viewModel.selectTag(tag)
                                },
                                label = { Text("#$tag") },
                                modifier = Modifier.height(24.dp)
                            )
                        }

                        document.metricValues.forEach { mv ->
                            AssistChip(
                                onClick = {},
                                enabled = false,
                                label = { Text("${mv.id} = ${mv.value}") },
                                leadingIcon = {
                                    Icon(
                                        FontAwesomeIcons.Regular.ChartBar,
                                        contentDescription = "Metric Icon",
                                        modifier = Modifier.Companion.size(12.dp)
                                    )
                                },
                                modifier = Modifier.height(24.dp)
                            )
                        }
                    }
                }
            }

            Box {
                IconButton(onClick = { }, enabled = false) {
                    Icon(
                        imageVector = FontAwesomeIcons.Solid.EllipsisV,
                        contentDescription = "Options",
                        modifier = Modifier.size(14.dp)
                    )
                }
            }
        }
    }
}