package xyz.lepisma.harp.screens

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material3.AssistChip
import androidx.compose.material3.AssistChipDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.solid.Filter
import xyz.lepisma.harp.viewmodel.ProfileViewModel

/**
 * Tag selection bar, most of the state is maintained outside of it
 */
@Composable
fun TagsSelector(tags: List<String>, viewModel: ProfileViewModel) {
    val selectedTags by viewModel.selectedTags.collectAsState()
    var expanded by remember { mutableStateOf(false) }

    LaunchedEffect(selectedTags) {
        // We only allow showing the tags view during tag selection, not hiding it.
        if (selectedTags.isNotEmpty()) {
            expanded = true
        }
    }

    Column(
        modifier = Modifier
            .fillMaxWidth()
            .background(color = MaterialTheme.colorScheme.surfaceVariant)
            .border(1.dp, MaterialTheme.colorScheme.outline.copy(alpha = 0.06f))
            .padding(vertical = 8.dp, horizontal = 20.dp)
    ) {
        TextButton(
            onClick = { expanded = !expanded },
            modifier = Modifier
                //.padding(vertical = 0.dp)
                .heightIn(min = 0.dp),
            contentPadding = PaddingValues(4.dp)
        ) {
            Row(
                verticalAlignment = Alignment.CenterVertically,
            ) {
                Icon(
                    imageVector = FontAwesomeIcons.Solid.Filter,
                    contentDescription = "Filter icon",
                    modifier = Modifier.size(10.dp),
                    tint = Color.Gray
                )
                Text(
                    "Filter by tags",
                    style = MaterialTheme.typography.labelMedium,
                    modifier = Modifier.padding(start = 4.dp)
                )
            }
        }

        AnimatedVisibility(
            visible = expanded
        ) {
            FlowRow(
                horizontalArrangement = Arrangement.spacedBy(6.dp),
                verticalArrangement = Arrangement.spacedBy(6.dp),
                modifier = Modifier.padding(bottom = 8.dp)
            ) {
                tags.forEachIndexed { index, tag ->
                    val selected = selectedTags.contains(tag)

                    AssistChip(
                        onClick = { viewModel.toggleTag(tag) },
                        label = { Text("#$tag", style = MaterialTheme.typography.labelMedium) },
                        modifier = Modifier
                            .height(24.dp)
                            .padding(0.dp),
                        colors = AssistChipDefaults.assistChipColors(
                            containerColor = if (selected) MaterialTheme.colorScheme.primary else MaterialTheme.colorScheme.surfaceVariant,
                            labelColor = if (selected) MaterialTheme.colorScheme.onPrimary else MaterialTheme.colorScheme.onSurfaceVariant
                        ),
                        border = if (selected) null else AssistChipDefaults.assistChipBorder(true)
                    )
                }
            }
        }
    }
}