package xyz.lepisma.harp.screens

import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.regular.File
import io.github.vinceglb.filekit.FileKit
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.dialogs.openFileWithDefaultApplication
import xyz.lepisma.harp.data.Asset
import xyz.lepisma.harp.data.resolveUriSubpath
import xyz.lepisma.harp.data.toastNotify
import xyz.lepisma.harp.viewmodel.ProfileViewModel

/**
 * Return file mapping to the asset using org-attachment sub path style
 */
private fun assetFilePath(asset: Asset, parentUuid: String, dataDir: PlatformFile): PlatformFile? =
    dataDir.resolveUriSubpath(parentUuid.take(2))
        ?.resolveUriSubpath(parentUuid.drop(2))
        ?.resolveUriSubpath(asset.fileName)

@Composable
fun AssetChip(asset: Asset, parentUuid: String, profileViewModel: ProfileViewModel) {
    val dataDir by profileViewModel.dataDirPath.collectAsState()

    Row(
        verticalAlignment = Alignment.CenterVertically,
        modifier = Modifier
            .padding(2.dp)
            .background(
                MaterialTheme.colorScheme.surfaceVariant,
                MaterialTheme.shapes.small
            )
            .border(
                1.dp,
                MaterialTheme.colorScheme.outline.copy(alpha = 0.6f),
                MaterialTheme.shapes.small
            )
            .padding(horizontal = 8.dp, vertical = 4.dp)
            .clickable {
                if (dataDir == null) {
                    toastNotify("Data directory not set for this profile, unable to work with attachments.")
                } else {
                    val assetPath = assetFilePath(asset, parentUuid, dataDir!!)
                    if (assetPath == null) {
                        toastNotify("Unable to resolve asset file path")
                    } else {
                        FileKit.openFileWithDefaultApplication(assetPath)
                    }
                }
            }
    ) {
        Icon(
            imageVector = FontAwesomeIcons.Regular.File,
            contentDescription = null,
            tint = MaterialTheme.colorScheme.outline,
            modifier = Modifier.size(MaterialTheme.typography.titleMedium.fontSize.value.dp)
        )

        Text(
            text = asset.fileName,
            style = MaterialTheme.typography.titleMedium,
            color = MaterialTheme.colorScheme.secondary,
            fontFamily = FontFamily.Monospace,
            maxLines = 1,
            overflow = TextOverflow.MiddleEllipsis,
            modifier = Modifier.padding(start = 5.dp)
        )
    }
}