package xyz.lepisma.harp.data

import xyz.lepisma.orgmode.OrgDocument
import xyz.lepisma.orgmode.OrgSection
import kotlin.uuid.ExperimentalUuidApi

enum class ProfileType {
    HUMAN, CAT, DOG
}

data class Profile(
    val uuid: String,
    val name: String,
    val metadata: Metadata,
    val journals: List<Journal>,
    val documents: List<Document>,
    val reminders: List<Reminder>,
    val type: ProfileType = ProfileType.HUMAN
)

/**
 * Return all unique tags from journal, documents, and everywhere else possible
 */
fun Profile.tags(): List<String> {
    val tags = mutableSetOf<String>()

    this.journals.forEach { journal ->
        journal.entries.forEach { entry ->
            entry.tags.forEach { tags.add(it) }
        }
    }

    this.documents.forEach { document ->
        document.tags.forEach { tags.add(it) }
    }

    this.metadata.metrics.forEach { metric ->
        metric.tags.forEach { tags.add(it) }
    }

    this.reminders.forEach { reminder ->
        reminder.tags.forEach { tags.add(it) }
    }

    // This is a special tag that tells whether an item has attachment
    tags.remove("ATTACH")

    return tags.toList().sorted()
}

fun fromOrgDocument(document: OrgDocument): Result<Profile> {
    val journalsSection = document.content.find { orgLineToString(it.heading.title).trim() == "Journals" }
        ?: return Result.failure(Exception("No Journals section found in document"))
    val documentsSection = document.content.find { orgLineToString(it.heading.title).trim() == "Documents" }
        ?: return Result.failure(Exception("No Documents section found in document"))
    val metadataSection = document.content.find { orgLineToString(it.heading.title).trim() == "Metadata" }
        ?: return Result.failure(Exception("No Metadata section found in document"))

    // TODO: Make this section mandatory also
    val remindersSection = document.content.find { orgLineToString(it.heading.title).trim() == "Reminders" }

    val idLine = document.preamble.properties?.map?.get("ID")
        ?: return Result.failure(Exception("No ID found for the document"))

    val typeLine = document.preamble.properties?.map?.get("HARP_TYPE")
    val profileType = if (typeLine != null) {
        when (orgLineToString(typeLine).trim().lowercase()) {
            "cat" -> ProfileType.CAT
            "dog" -> ProfileType.DOG
            else -> ProfileType.HUMAN
        }
    } else {
        ProfileType.HUMAN
    }

    // Name could be set as HARP_NAME property of the document, falling back to title as the backup
    val nameLine = document.preamble.properties?.map?.get("HARP_NAME")
        ?: document.preamble.title

    val journals = journalsSection.body.filter { it is OrgSection }.mapNotNull { parseJournal(it as OrgSection) }
    if (journals.isEmpty()) {
        // There must be at least one journal, if we start from the app itself that would be 'Main'
        return Result.failure(Exception("No journal found under Journals"))
    }

    val documents = documentsSection.body.filter { it is OrgSection }.mapNotNull { parseDocumentEntry(it as OrgSection) }

    val reminders = remindersSection?.body?.filter { it is OrgSection }?.mapNotNull { parseReminder(it as OrgSection) }
        ?: emptyList()

    val metadata = parseMetadata(metadataSection)

    return Result.success(Profile(
        uuid = orgLineToString(idLine).trim(),
        name = orgLineToString(nameLine).trim(),
        metadata = metadata,
        journals = journals,
        documents = documents,
        reminders = reminders,
        type = profileType
    ))
}

@OptIn(ExperimentalUuidApi::class)
fun newProfileText(name: String, uuid: String): String {
    return """
        |:PROPERTIES:
        |:ID: $uuid
        |:END:
        |#+TITLE: $name
        |
        |* Metadata
        |** Sources
        |
        |** Metrics
        |
        |* Journals
        |** Main
        |
        |* Documents
        |
    """.trimMargin()
}