package xyz.lepisma.harp.data

import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.format
import kotlinx.datetime.format.char
import xyz.lepisma.harp.screens.metrics.MetricsView
import xyz.lepisma.orgmode.OrgChunk
import xyz.lepisma.orgmode.OrgHeading
import xyz.lepisma.orgmode.OrgHeadingLevel
import xyz.lepisma.orgmode.OrgInlineElem
import xyz.lepisma.orgmode.OrgLine
import xyz.lepisma.orgmode.OrgProperties
import xyz.lepisma.orgmode.OrgSection
import xyz.lepisma.orgmode.OrgTags
import xyz.lepisma.orgmode.core.ParsingResult
import xyz.lepisma.orgmode.lexer.OrgLexer
import xyz.lepisma.orgmode.lexer.Token
import xyz.lepisma.orgmode.parseSection
import kotlin.uuid.Uuid

data class JournalEntry(
    val datetime: LocalDateTime,
    val uuid: String,
    val tags: List<String>,
    val metricValues: List<MetricValue>,
    val text: String,
    val assets: List<Asset>,
    val isPrivate: Boolean
)

fun JournalEntry.formatString(): String {
    val formatter = LocalDateTime.Format {
        char('<')
        year()
        char('-')
        monthNumber()
        char('-')
        dayOfMonth()
        char(' ')
        hour()
        char(':')
        minute()
        char('>')
    }

    val tagsString = if (this.tags.isNotEmpty()) {
        ":" + this.tags.joinToString { ":" } + ":"
    } else {
        ""
    }

    return """
        |*** Entry    $tagsString
        |:PROPERTIES:
        |:ID: ${this.uuid}
        |:DATETIME: ${this.datetime.format(formatter)}
        |:PRIVATE: nil
        |:END:
        |${this.text}
    """.trimMargin()
}

fun parseJournalEntry(section: OrgSection): JournalEntry? {
    // The format enforces a certain nesting structure
    if (section.heading.level.level != 3) {
        return null
    }

    val props = section.heading.properties?.map
    if (props == null) {
        return null
    }

    if (!props.contains("ID") || !props.contains("DATETIME")) {
        return null
    }
    val id = orgLineToString(props["ID"]!!).trim()

    // HACK: Prop values are parsed as plain text for now so we will invoke the datetime parser manually
    // till the newer parsers handle the plain text well
    val dtTokens = OrgLexer(props["DATETIME"]?.items[0]?.tokens[0]?.text ?: "").tokenize()
    if (dtTokens.size != 3) {
        // Tokens will be SOF, DatetimeStamp, and EOF
        return null
    }
    val dtStamp = dtTokens[1] as Token.DatetimeStamp
    val dt = LocalDateTime(
        year = dtStamp.date.year,
        monthNumber = dtStamp.date.monthNumber,
        dayOfMonth = dtStamp.date.dayOfMonth,
        hour = dtStamp.time?.first?.hour ?: 0,
        minute = dtStamp.time?.first?.minute ?: 0,
        second = dtStamp.time?.first?.second ?: 0
    )

    // TODO:
    // Parse assets list from tag, body, and attachment dir (not present now)
    // Parse isPrivate from prop

    val body = section.body.joinToString("") { it.tokens.joinToString("") { it.text } }

    val metricValues = mutableListOf<MetricValue>()
    val tagSet = (section.heading.tags?.tags ?: emptyList()).toMutableSet()

    section.body.forEach { chunk ->
        when (chunk) {
            // TODO: We only parse plain paragraph for this right now, finish it
            is OrgChunk.OrgParagraph -> chunk.items.forEach { it ->
                when (it) {
                    is OrgInlineElem.HashMetric -> {
                        metricValues.add(
                            MetricValue(
                                id = it.metric,
                                datetime = dt,
                                value = it.value.toFloat(),
                                reference = id
                            )
                        )
                    }
                    is OrgInlineElem.HashTag -> {
                        tagSet.add(it.text)
                    }
                    else -> {  }
                }
            }
            else -> { }
        }
    }

    return JournalEntry(
        uuid = id,
        datetime = dt,
        tags = tagSet.toList().sorted(),
        metricValues = metricValues,
        text = body,
        assets = emptyList(),
        isPrivate = false
    )
}