package xyz.lepisma.harp.data

import io.github.vinceglb.filekit.BookmarkData
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.bookmarkData
import io.github.vinceglb.filekit.fromBookmarkData
import kotlinx.serialization.Serializable
import xyz.lepisma.harp.Platform
import kotlin.io.encoding.Base64

/**
 * Represents a file under a directory represented as platform file.
 *
 * This is abstracted out since FileKit doesn't seem to work well with nesting for content type
 * URLs in Android.
 *
 * There are two ways this can represent a file:
 * 1. The user picks the file directly like when opening an existing profile
 * 2. The user picks the file by selecting root dir (usually when creating new profile) and then
 *    the app writes a new file under it. In this case we can only persist permission for `root` so
 *    we store root and documentId and recreate the filePath while deserializing the object.
 */
sealed interface FileUnderDir {
    val filePath: PlatformFile

    data class Direct(
        override val filePath: PlatformFile
    ) : FileUnderDir

    data class UnderRoot(
        val root: PlatformFile,
        val documentId: String,
        override val filePath: PlatformFile
    ) : FileUnderDir
}

@Serializable
sealed interface PersistedFileUnderDir {
    @Serializable
    data class Direct(
        val fileBookmarkData: String
    ) : PersistedFileUnderDir

    @Serializable
    data class UnderRoot(
        val rootBookmarkData: String,
        val documentId: String
    ) : PersistedFileUnderDir
}

suspend fun FileUnderDir.toPersistable(): PersistedFileUnderDir =
    when (this) {
        is FileUnderDir.Direct ->
            PersistedFileUnderDir.Direct(
                Base64.encode(filePath.bookmarkData().bytes)
            )

        is FileUnderDir.UnderRoot ->
            PersistedFileUnderDir.UnderRoot(
                Base64.encode(root.bookmarkData().bytes),
                documentId
            )
    }

fun PersistedFileUnderDir.toRuntime(): Result<FileUnderDir> =
    when (this) {
        is PersistedFileUnderDir.Direct -> {
            val file = PlatformFile.fromBookmarkData(Base64.decode(fileBookmarkData))
            Result.success(FileUnderDir.Direct(file))
        }

        is PersistedFileUnderDir.UnderRoot -> {
            val root = PlatformFile.fromBookmarkData(Base64.decode(rootBookmarkData))
            val file = resolveFileUnderDir(root, documentId)
                ?: return Result.failure(Exception("Unable to resolve file under dir"))

            Result.success(FileUnderDir.UnderRoot(root, documentId, file))
        }
    }

/**
 * Resolve URI based subpaths since that doesn't work with the default div operator in FileKit
 */
expect fun PlatformFile.resolveUriSubpath(sub: String): PlatformFile?
expect fun resolveFileUnderDir(root: PlatformFile, documentId: String): PlatformFile?
expect fun createFile(root: PlatformFile, fileName: String): FileUnderDir.UnderRoot?