package xyz.lepisma.harp.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.readString
import io.github.vinceglb.filekit.writeString
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.drop
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import xyz.lepisma.harp.data.FileUnderDir
import xyz.lepisma.harp.data.Journal
import xyz.lepisma.harp.data.JournalEntry
import xyz.lepisma.harp.data.Profile
import xyz.lepisma.harp.data.formatString
import xyz.lepisma.harp.data.fromOrgDocument
import xyz.lepisma.harp.data.orgLineToString
import xyz.lepisma.harp.data.toastNotify
import xyz.lepisma.orgmode.OrgDocument
import xyz.lepisma.orgmode.OrgSection
import xyz.lepisma.orgmode.lexer.OrgLexer
import xyz.lepisma.orgmode.parse

@OptIn(FlowPreview::class)
class ProfileViewModel : ViewModel() {
    private val _file = MutableStateFlow<FileUnderDir?>(null)
    private val _dataDirPath = MutableStateFlow<PlatformFile?>(null)
    private val _buffer = MutableStateFlow<String?>(null)

    val file = _file.asStateFlow()

    // Path to data directory for attachments and documents. If this is not present, the UI
    // disables certain capabilities and shows appropriate notices.
    val dataDirPath = _dataDirPath.asStateFlow()

    // Buffer maintains the currently opened file in plain string format
    val buffer = _buffer.asStateFlow()

    private val _doc: StateFlow<OrgDocument?> =
        buffer
            .map { text -> text?.let {
                try {
                    parse(OrgLexer(text).tokenize())
                } catch (_: Exception) {
                    toastNotify("Unable to parse the file as valid org mode file")
                    null
                }
            } }
            .stateIn(viewModelScope, SharingStarted.Eagerly, null)

    // We trigger profile parsing on every change in buffer
    val profile: StateFlow<Profile?> =
        _doc
            .map { d -> d?.let {
                val parsingResult = fromOrgDocument(it)
                if (parsingResult.isSuccess) {
                    parsingResult.getOrNull()
                } else {
                    toastNotify("Parsing error: ${parsingResult.exceptionOrNull()?.message}")
                    null
                }
            } }
            .stateIn(viewModelScope, SharingStarted.Eagerly, null)

    // On every change, debounce and write to file
    init {
        viewModelScope.launch {
            buffer
                .filterNotNull()
                // HACK: This is not correct when working across multiple profiles
                .drop(1)
                .debounce(500)
                .collect { text ->
                    withContext(Dispatchers.Default) {
                        _file.value?.filePath?.writeString(text)
                    }
                }
        }
    }

    // TODO: Check local file and revert buffer
    suspend fun loadProfile(file: FileUnderDir, dataDirPath: PlatformFile?): Profile {
        file.filePath.readString().let { text ->
            _file.value = file
            _buffer.value = text

            setDataDirPath(dataDirPath)
        }

        return profile.filterNotNull().first()
    }

    fun setDataDirPath(dataDirPath: PlatformFile?) {
        _dataDirPath.value = dataDirPath
    }

    private val _selectedTags = MutableStateFlow<Set<String>>(emptySet())
    val selectedTags = _selectedTags.asStateFlow()

    fun clearTagSelection() {
        _selectedTags.update { emptySet() }
    }

    fun selectTag(tag: String) {
        _selectedTags.update { it + tag }
    }

    fun unselectTag(tag: String) {
        _selectedTags.update { it - tag }
    }

    fun toggleTag(tag: String) {
        _selectedTags.update { current ->
            if (tag in current) current - tag else current + tag
        }
    }

    fun findJournalSectionByName(name: String): Result<OrgSection> {
        val doc = _doc.value
            ?: return Result.failure(IllegalStateException("Document is null"))

        val journalsSection =
            doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }
                ?: return Result.failure(NoSuchElementException("Unable to find Journals heading"))

        for (chunk in journalsSection.body) {
            if (chunk is OrgSection && chunk.heading.level.level == 2) {
                // This is a journal
                val title = orgLineToString(chunk.heading.title).trim()
                if (title == name) {
                    return Result.success(chunk)
                }
            }
        }

        return Result.failure(NoSuchElementException("Unable to find journal $name"))
    }

    /**
     * Delete an org section from the current buffer and file
     */
    private fun deleteSection(section: OrgSection): Result<Unit> {
        _doc.value?.let { doc ->
            val range = section.tokens.first().range.first to section.tokens.last().range.second

            if (_buffer.value == null) {
                return Result.failure(IllegalStateException("Buffer not loaded"))
            }

            val newBuffer = _buffer.value?.let { text ->
                text.take(range.first) + text.substring(range.second)
            }

            _buffer.value = newBuffer
        }
        return Result.success(Unit)
    }

    fun deleteJournalByName(name: String) {
        val journalSection = findJournalSectionByName(name).getOrElse {
            toastNotify("Unable to find journal $name")
            return
        }

        deleteSection(journalSection).onFailure { e ->
            toastNotify(e.message ?: "Failed to delete journal $name")
        }
    }

    /**
     * Add child just under parent's heading
     */
    private fun addUnder(parent: OrgSection, childText: String): Result<Unit> {
        if (_doc.value == null) {
            return Result.failure(IllegalStateException("Document not loaded"))
        }

        val pos = parent.heading.tokens.last().range.second
        val buffer = _buffer.value
            ?: return Result.failure(IllegalStateException("Buffer not loaded"))
        _buffer.value = buffer.take(pos) + "\n" + childText + "\n\n" + buffer.substring(pos)

        return Result.success(Unit)
    }

    fun addJournal(journalName: String) {
        findJournalSectionByName(journalName).onSuccess {
            toastNotify("Journal '$journalName' already exists, pick another name")
            return
        }

        val journalText = Journal(
            name = journalName,
            entries = emptyList()
        ).formatString()

        val doc = _doc.value
        if (doc == null) {
            toastNotify("Document not loaded")
            return
        }

        val journalsSection = doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }

        if (journalsSection == null) {
            toastNotify("Unable to find Journals heading")
            return
        }

        addUnder(journalsSection, journalText).onFailure { exception ->
            toastNotify(exception.message ?: "Unable to add journal $journalName")
        }
    }

    /**
     * Change the title of the section to the provided one
     */
    private fun renameSection(section: OrgSection, newName: String): Result<Unit> {
        if (_doc.value == null) {
            return Result.failure(IllegalStateException("Document not loaded"))
        }

        val beg = section.heading.title.tokens.first().range.first
        val end = section.heading.title.tokens.last().range.second

        val buffer = _buffer.value
            ?: return Result.failure(IllegalStateException("Buffer not loaded"))
        _buffer.value = buffer.take(beg) + newName + buffer.substring(end)

        return Result.success(Unit)
    }

    fun renameJournal(journalName: String, newName: String) {
        val oldJournal = findJournalSectionByName(journalName).getOrElse {
            toastNotify("Journal '$journalName' not found")
            return
        }

        renameSection(oldJournal, newName).onFailure { exception ->
            toastNotify(exception.message ?: "Unable to rename journal '$journalName")
        }
    }

    fun deleteJournal(journalName: String) {
        val journal = findJournalSectionByName(journalName).getOrElse {
            toastNotify("Journal '$journalName' not found")
            return
        }

        deleteSection(journal).onFailure { exception ->
            toastNotify(exception.message ?: "Unable to delete journal '$journalName")
        }
    }

    fun findJournalEntrySectionById(entryId: String): OrgSection? {
        return _doc.value?.let { doc ->
            val journalsSection =
                doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }

            if (journalsSection == null) {
                toastNotify("Unable to find Journals heading")
                return null
            }

            // We find the entry across all journals. This can be narrowed down by journal but we
            // don't maintain current journal context in view model so we will go with this hack for
            // now
            var entrySection: OrgSection? = null
            for (chunk in journalsSection.body) {
                if (chunk is OrgSection && chunk.heading.level.level == 2) {
                    entrySection = chunk.body.find { subchunk ->
                        if (subchunk is OrgSection && subchunk.heading.level.level == 3) {
                            val id = subchunk.heading.properties?.map?.get("ID")?.let {
                                orgLineToString(it).trim()
                            }
                            id != null && id == entryId
                        } else {
                            false
                        }
                    } as OrgSection?

                    if (entrySection != null) {
                        break
                    }
                }
            }

            if (entrySection == null) {
                toastNotify("Unable to find section with id $entryId")
            }

            entrySection
        }
    }

    fun deleteJournalEntry(entry: JournalEntry) {
        val entrySection = findJournalEntrySectionById(entry.uuid)

        if (entrySection == null) {
            toastNotify("Unable to delete journal entry")
            return
        }

        deleteSection(entrySection).onFailure { e ->
            toastNotify(e.message ?: "Failed to delete journal entry")
        }
    }

    fun addJournalEntry(journalName: String, entry: JournalEntry) {
        val entryText = entry.formatString()

        val doc = _doc.value
        if (doc == null) {
            toastNotify("Document not loaded")
            return
        }

        val journalsSection = doc.content.find { orgLineToString(it.heading.title).trim() == "Journals" }

        if (journalsSection == null) {
            toastNotify("Unable to find Journals heading")
            return
        }

        val journal = journalsSection.body.find {
            it is OrgSection && orgLineToString(it.heading.title).trim() == journalName
        } as OrgSection?

        if (journal == null) {
            toastNotify("Unable to find journal $journalName")
            return
        }

        addUnder(journal, entryText).onFailure { exception ->
            toastNotify(exception.message ?: "Unable to add entry")
        }
    }
}