package xyz.lepisma.harp.screens.metrics

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.AssistChip
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.solid.ArrowRight
import compose.icons.fontawesomeicons.solid.CaretDown
import compose.icons.fontawesomeicons.solid.CaretRight
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.format
import kotlinx.datetime.format.DayOfWeekNames
import kotlinx.datetime.format.MonthNames
import kotlinx.datetime.format.char
import xyz.lepisma.harp.data.Metric
import xyz.lepisma.harp.data.MetricValue
import xyz.lepisma.harp.data.Profile
import xyz.lepisma.harp.screens.DTStampInlineView
import xyz.lepisma.harp.viewmodel.ProfileViewModel


@Composable
fun MetricsView(profile: Profile, viewModel: ProfileViewModel) {
    // Map between metric id and list of metric values for that id
    val metricMap: MutableMap<String, MutableList<MetricValue>> = mutableMapOf()
    val selectedTags by viewModel.selectedTags.collectAsState()

    profile.journals.forEach { journal ->
        journal.entries.forEach { entry ->
            entry.metricValues.forEach { mv ->
                if (!metricMap.contains(mv.id)) {
                    metricMap[mv.id] = mutableListOf<MetricValue>()
                }

                metricMap[mv.id]?.add(mv)
            }
        }
    }

    profile.documents.forEach { document ->
        document.metricValues.forEach { mv ->
            if (!metricMap.contains(mv.id)) {
                metricMap[mv.id] = mutableListOf<MetricValue>()
            }

            metricMap[mv.id]?.add(mv)
        }
    }

    val metricMetadataMap: MutableMap<String, Metric> = mutableMapOf()
    profile.metadata.metrics.map {
        metricMetadataMap[it.id] = it
    }

    val selectedMetricMap = if (selectedTags.isEmpty()) {
        metricMap
    } else {
        metricMap.filter {
            if (metricMetadataMap.contains(it.key)) {
                metricMetadataMap[it.key]?.tags?.any { selectedTags.contains(it) } ?: false
            } else {
                // In case of tag selection, we skip all untagged, no metadata, metrics
                false
            }
        }
    }

    Text(
        "This section displays metrics logged in journal and document entries",
        style = MaterialTheme.typography.bodyMedium,
        modifier = Modifier.padding(vertical = 8.dp)
    )

    Text(
        "Total ${selectedMetricMap.size} of ${metricMap.size} metrics displayed",
        style = MaterialTheme.typography.labelMedium,
        modifier = Modifier.padding(vertical = 8.dp)
    )

    LazyColumn {
        items(selectedMetricMap.entries.toList()) { kv ->
            val metric = metricMetadataMap[kv.key]

            ElevatedCard(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(vertical = 8.dp)
            ) {
                Column (
                    modifier = Modifier
                        .padding(14.dp)
                ) {
                    Text(
                        text = metric?.name ?: kv.key,
                        style = MaterialTheme.typography.titleLarge,
                        modifier = Modifier.padding(bottom = 10.dp)
                    )

                    val mvs = kv.value.sortedBy { it.datetime }

                    Row(
                        verticalAlignment = Alignment.Bottom
                    ) {
                        Text(
                            text = "${mvs.last().value} ${metric?.unit ?: ""}",
                            style = MaterialTheme.typography.headlineLarge,
                        )
                        Text(
                            text = "as of ${mvs.last().datetime.format(LocalDateTime.Format {
                                dayOfWeek(DayOfWeekNames.ENGLISH_ABBREVIATED)
                                char(' ')
                                monthName(MonthNames.ENGLISH_ABBREVIATED)
                                char(' ')
                                dayOfMonth()
                                chars(", ")
                                year()
                            })}",
                            fontStyle = FontStyle.Italic,
                            modifier = Modifier.padding(start = 5.dp)
                        )
                    }

                    if (mvs.size > 1) {
                        MetricsPlot(
                            mvs = mvs,
                            metric = metric,
                            modifier = Modifier
                                .fillMaxWidth()
                                .height(200.dp)
                                .padding(8.dp)
                        )
                    }

                    HorizontalDivider(modifier = Modifier.padding(vertical = 10.dp))

                    var expanded by remember { mutableStateOf(false) }

                    Column(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(vertical = 10.dp)
                    ) {
                        metric?.tags?.takeIf { it.isNotEmpty() }?.let { tags ->
                            FlowRow(
                                horizontalArrangement = Arrangement.spacedBy(6.dp),
                                verticalArrangement = Arrangement.spacedBy(6.dp),
                                modifier = Modifier
                                    .padding(bottom = 10.dp)
                            ) {
                                tags.forEach { tag ->
                                    AssistChip(
                                        onClick = {
                                            viewModel.clearTagSelection()
                                            viewModel.selectTag(tag)
                                        },
                                        label = { Text("#$tag", style = MaterialTheme.typography.labelMedium) },
                                        modifier = Modifier
                                            .height(24.dp)
                                            .padding(0.dp)
                                    )
                                }
                            }
                        }

                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            modifier = Modifier
                                .fillMaxWidth()
                                .clickable { expanded = !expanded }
                        ) {
                            Text(
                                text = "All entries",
                                style = MaterialTheme.typography.titleMedium,
                                modifier = Modifier.weight(1f)
                            )

                            Icon(
                                imageVector = if (expanded) FontAwesomeIcons.Solid.CaretDown else FontAwesomeIcons.Solid.CaretRight,
                                contentDescription = null,
                                modifier = Modifier
                                    .size(20.dp)
                                    .padding(start = 10.dp)
                            )
                        }

                        AnimatedVisibility(expanded) {
                            Column(
                                modifier = Modifier.padding(top = 15.dp)
                            ) {
                                mvs.reversed().forEach { mv ->
                                    Row(
                                        verticalAlignment = Alignment.CenterVertically,
                                        modifier = Modifier.padding(vertical = 5.dp)
                                    ) {
                                        DTStampInlineView(mv.datetime)
                                        Icon(
                                            FontAwesomeIcons.Solid.ArrowRight,
                                            contentDescription = null,
                                            modifier = Modifier
                                                .padding(horizontal = 6.dp)
                                                .size(12.dp)
                                        )
                                        Text(
                                            "${mv.value} ${metric?.unit ?: ""}",
                                            fontWeight = FontWeight.Bold,
                                            fontFamily = FontFamily.Monospace
                                        )
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}