package xyz.lepisma.harp.screens.journals

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.MenuAnchorType
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.regular.Edit
import compose.icons.fontawesomeicons.regular.PlusSquare
import compose.icons.fontawesomeicons.regular.TrashAlt
import compose.icons.fontawesomeicons.solid.EllipsisV
import xyz.lepisma.harp.data.Journal
import xyz.lepisma.harp.data.toastNotify
import xyz.lepisma.harp.viewmodel.ProfileViewModel

private enum class JournalOp {
    NEW, EDIT
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun JournalView(
    journals: List<Journal>,
    viewModel: ProfileViewModel,
    onJournalIdxChange: (Int) -> Unit,
    modifier: Modifier = Modifier
) {
    var selectedJournalIdx: Int by remember { mutableStateOf(0) }
    val journal = journals[selectedJournalIdx]

    var selectorExpanded by remember { mutableStateOf(false) }
    var journalMenuExpanded by remember { mutableStateOf(false) }
    val selectedTags by viewModel.selectedTags.collectAsState()

    // Variable to hold name of journal under creation or edit
    var journalName: String by remember { mutableStateOf("") }
    var journalOp: JournalOp by remember { mutableStateOf(JournalOp.NEW) }
    var showJournalNameDialog by remember { mutableStateOf(false) }

    var showDeleteConfirmation by remember { mutableStateOf(false) }

    Column(
        modifier = modifier.fillMaxSize()
    ) {
        Text(
            "This section displays text entries organized under different journals",
            style = MaterialTheme.typography.bodyMedium,
            modifier = Modifier.padding(vertical = 8.dp)
        )
        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.fillMaxWidth()
        ) {
            ExposedDropdownMenuBox(
                expanded = selectorExpanded,
                onExpandedChange = { selectorExpanded = !selectorExpanded },
                modifier = Modifier
                    .weight(1f)
                    .padding(bottom = 14.dp)
            ) {
                TextField(
                    value = journal.name,
                    onValueChange = { },
                    readOnly = true,
                    label = { Text("Selected Journal") },
                    trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(selectorExpanded) },
                    colors = TextFieldDefaults.colors(
                        focusedContainerColor = Color.Transparent,
                        unfocusedContainerColor = Color.Transparent,
                        disabledContainerColor = Color.Transparent,
                    ),
                    modifier = Modifier
                        .fillMaxWidth()
                        .menuAnchor(MenuAnchorType.PrimaryEditable, true)
                )

                ExposedDropdownMenu(
                    expanded = selectorExpanded,
                    onDismissRequest = { selectorExpanded = false }
                ) {
                    journals.forEachIndexed { idx, journal ->
                        DropdownMenuItem(
                            text = { Text(journal.name) },
                            onClick = {
                                selectedJournalIdx = idx
                                selectorExpanded = false
                                onJournalIdxChange(idx)
                            }
                        )
                    }
                }
            }

            Spacer(Modifier.width(8.dp))

            Box {
                IconButton(
                    onClick = { journalMenuExpanded = true }
                ) {
                    Icon(
                        imageVector = FontAwesomeIcons.Solid.EllipsisV,
                        contentDescription = "Journal action menu",
                        modifier = Modifier.size(24.dp)
                    )
                }
                DropdownMenu(
                    expanded = journalMenuExpanded,
                    onDismissRequest = { journalMenuExpanded = false }
                ) {
                    DropdownMenuItem(
                        text = { Text("Rename Journal") },
                        onClick = {
                            journalOp = JournalOp.EDIT
                            journalName = journal.name
                            showJournalNameDialog = true
                            journalMenuExpanded = false
                        },
                        leadingIcon = {
                            Icon(
                                FontAwesomeIcons.Regular.Edit,
                                "Rename",
                                modifier = Modifier.size(12.dp)
                            )
                        }
                    )
                    DropdownMenuItem(
                        text = { Text("Create New Journal") },
                        onClick = {
                            journalOp = JournalOp.NEW
                            journalName = ""
                            showJournalNameDialog = true
                            journalMenuExpanded = false
                        },
                        leadingIcon = {
                            Icon(
                                FontAwesomeIcons.Regular.PlusSquare,
                                "Create",
                                modifier = Modifier.size(12.dp)
                            )
                        }
                    )
                    DropdownMenuItem(
                        text = { Text("Delete Journal") },
                        onClick = {
                            showDeleteConfirmation = true
                            journalMenuExpanded = false
                        },
                        leadingIcon = {
                            Icon(
                                FontAwesomeIcons.Regular.TrashAlt,
                                "Delete",
                                modifier = Modifier.size(12.dp)
                            )
                        })
                }
            }

            if (showJournalNameDialog) {
                AlertDialog(
                    onDismissRequest = { showJournalNameDialog = false},
                    confirmButton = {
                        TextButton(
                            onClick = {
                                when (journalOp) {
                                    JournalOp.EDIT -> viewModel.renameJournal(journal.name, journalName)
                                    // Switch to new journal happens automatically since the journal is
                                    // added just after the heading, and re-parsing takes the first
                                    // journal as the primary one(?).
                                    JournalOp.NEW -> viewModel.addJournal(journalName)
                                }
                                showJournalNameDialog = false
                            },
                            enabled = journalName.isNotEmpty()
                        ) {
                            Text("Confirm")
                        }
                    },
                    title = {
                        Text( when (journalOp) {
                            JournalOp.EDIT -> "Rename Journal"
                            JournalOp.NEW -> "Create New Journal"
                        })
                    },
                    text = {
                        Column {
                            TextField(
                                value = journalName,
                                label = { Text("Journal name") },
                                onValueChange = { journalName = it }
                            )
                        }
                    }
                )
            }

            if (showDeleteConfirmation) {
                if (journals.size == 1) {
                    toastNotify("Can't delete the only journal")
                    showDeleteConfirmation = false
                } else {
                    AlertDialog(
                        onDismissRequest = { showDeleteConfirmation = false },
                        title = { Text("Delete Journal '${journal.name}'?") },
                        text = {
                            Text("Are your sure you want to delete the journal '${journal.name}'? This action cannot be undone.")
                        },
                        confirmButton = {
                            TextButton(onClick = {
                                viewModel.deleteJournal(journal.name)
                                showDeleteConfirmation = false
                            }) {
                                Text("Delete")
                            }
                        },
                        dismissButton = {
                            TextButton(onClick = { showDeleteConfirmation = false }) {
                                Text("Cancel")
                            }
                        }
                    )
                }
            }
        }

        val selectedEntries = if (selectedTags.isEmpty()) {
            journal.entries
        } else {
            journal.entries.filter { entry ->
                entry.tags.any { selectedTags.contains(it) }
            }
        }

        Text(
            "Total ${selectedEntries.size} of ${journal.entries.size} entries displayed",
            style = MaterialTheme.typography.labelMedium,
            modifier = Modifier.padding(vertical = 8.dp)
        )

        LazyColumn(modifier = Modifier.fillMaxSize()) {
            items(selectedEntries) { entry ->
                JournalEntryView(entry, viewModel = viewModel, modifier = Modifier.padding(vertical = 3.dp))
            }
        }
    }
}