package xyz.lepisma.harp.screens.home

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.dialogs.compose.rememberDirectoryPickerLauncher
import io.github.vinceglb.filekit.dialogs.compose.rememberFilePickerLauncher
import io.github.vinceglb.filekit.name
import kotlinx.coroutines.launch
import xyz.lepisma.harp.data.ProfileInfo
import xyz.lepisma.harp.data.readProfileInfo
import xyz.lepisma.harp.data.toastNotify

private enum class Step {
    INTRO_FILE,
    PICK_FILE,
    INTRO_DIR,
    PICK_DIR,
    DONE
}

/**
 * Ask user to pick the profile file, optionally a data directory, and return profileInfo
 */
@Composable
fun OpenProfileDialog(
    onCompletion: (ProfileInfo) -> Unit,
    onDismiss: () -> Unit
) {
    var step by remember { mutableStateOf(Step.INTRO_FILE) }
    var profileFile: PlatformFile? by remember { mutableStateOf(null) }
    var dataDirPath: PlatformFile? by remember { mutableStateOf(null) }
    val scope = rememberCoroutineScope()

    val dataDirPickerLauncher = rememberDirectoryPickerLauncher(
        title = "Data directory picker"
    ) { directory ->
        if (directory == null) {
            toastNotify("Unable to pick data directory")
            step = Step.DONE
            onDismiss()
            return@rememberDirectoryPickerLauncher
        }

        if (directory.name != "data") {
            toastNotify("Unable to work with data directory named ${directory.name}")
            step = Step.DONE
            onDismiss()
            return@rememberDirectoryPickerLauncher
        }

        dataDirPath = directory

        step = Step.DONE
    }

    val filePickerLauncher = rememberFilePickerLauncher(
        title = "Profile file picker"
    ) { file ->
        if (file == null) {
            toastNotify("Unable to pick profile org file")
            step = Step.DONE
            onDismiss()
            return@rememberFilePickerLauncher
        }
        profileFile = file
        step = Step.INTRO_DIR
    }

    when (step) {
        Step.INTRO_FILE -> AlertDialog(
            onDismissRequest = onDismiss,
            title = { Text("Profile file") },
            text = { Text("You will be asked to select the profile Org file once you press continue.") },
            confirmButton = {
                TextButton(onClick = { step = Step.PICK_FILE }) {
                    Text("Continue")
                }
            }
        )

        Step.PICK_FILE -> LaunchedEffect(Unit) {
            filePickerLauncher.launch()
        }

        Step.INTRO_DIR -> AlertDialog(
            onDismissRequest = onDismiss,
            title = { Text("Data directory") },
            text = {
                Column {
                    Text("Optionally select the folder named 'data' for this profile in the same directory as the Org file. If it doesn’t exist, create it first.")
                    Text("You can also do this later.", modifier = Modifier.padding(top = 4.dp))
                }
            },
            confirmButton = {
                TextButton(onClick = { step = Step.DONE }) {
                    Text("Skip")
                }
                TextButton(onClick = { step = Step.PICK_DIR }) {
                    Text("Continue")
                }
            }
        )

        Step.PICK_DIR -> LaunchedEffect(Unit) {
            dataDirPickerLauncher.launch()
        }

        Step.DONE -> {
            scope.launch {
                val profileInfo = readProfileInfo(profileFile!!, dataDirPath).getOrElse { exception ->
                    toastNotify(exception.message ?: "Error in reading profile info")
                    return@launch
                }

                onCompletion(profileInfo)
            }
        }
    }
}