package xyz.lepisma.harp.screens.home

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.dialogs.compose.rememberDirectoryPickerLauncher
import io.github.vinceglb.filekit.writeString
import kotlinx.coroutines.launch
import xyz.lepisma.harp.data.ProfileInfo
import xyz.lepisma.harp.data.createFile
import xyz.lepisma.harp.data.newProfileText
import xyz.lepisma.harp.data.toastNotify
import kotlin.uuid.ExperimentalUuidApi
import kotlin.uuid.Uuid


/**
 * Create new profile file under given directory and return the full path to it
 */
@OptIn(ExperimentalUuidApi::class)
suspend fun createNewProfile(dir: PlatformFile, name: String): ProfileInfo? {
    val fileName = "harp-profile-${name}.org"
    val fileUnderDir = createFile(dir, fileName)
        ?: return null

    try {
        val uuid = Uuid.random().toHexDashString()
        fileUnderDir.filePath.writeString(newProfileText(name, uuid))

        return ProfileInfo(
            uuid = uuid,
            name = name,
            profileFile = fileUnderDir,
            //  We do not create data directory and ask the user to choose or create it later
            //  in the workflow.
            dataDir = null
        )
    } catch (_: Exception) {
        return null
    }
}

@Composable
fun NewProfileForm(onCompletion: (ProfileInfo) -> Unit, onDismiss: () -> Unit) {
    var nameInput by remember { mutableStateOf("") }
    val scope = rememberCoroutineScope()

    val newDirWorkflowLauncher = rememberDirectoryPickerLauncher { dir ->
        if (dir == null) {
            toastNotify("Unable to pick directory for profile")
            return@rememberDirectoryPickerLauncher
        }

        scope.launch {
            val profileInfo = createNewProfile(dir, nameInput)

            if (profileInfo == null) {
                toastNotify("Unable to create file for $nameInput")
                return@launch
            }
            onCompletion(profileInfo)
        }
    }

    val canConfirm = nameInput.isNotBlank()
    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Create New Profile") },
        text = {
            Column {
                TextField(
                    value = nameInput,
                    onValueChange = { nameInput = it },
                    label = { Text("Profile Name") },
                    singleLine = true,
                    modifier = Modifier.padding(vertical = 10.dp)
                )

                Text(
                    text = if (nameInput.isNotBlank()) {
                        "A new file with name '$nameInput.harp.org'  will be created in the directory you select after pressing Create."
                    } else {
                        "A new file will be created in the directory you select after pressing Create."
                    },
                    modifier = Modifier.padding(top = 10.dp)
                )
            }
        },
        confirmButton = {
            TextButton(
                onClick = {
                    newDirWorkflowLauncher.launch()
                },
                enabled = canConfirm
            ) {
                Text("Create")
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text("Cancel")
            }
        }
    )
}