package xyz.lepisma.harp.screens

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.NavigationBar
import androidx.compose.material3.NavigationBarItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.unit.dp
import cafe.adriel.voyager.core.screen.Screen
import cafe.adriel.voyager.navigator.LocalNavigator
import cafe.adriel.voyager.navigator.currentOrThrow
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.regular.Bell
import compose.icons.fontawesomeicons.regular.ChartBar
import compose.icons.fontawesomeicons.regular.StickyNote
import compose.icons.fontawesomeicons.solid.Cat
import compose.icons.fontawesomeicons.solid.Dog
import compose.icons.fontawesomeicons.solid.NotesMedical
import compose.icons.fontawesomeicons.solid.Plus
import compose.icons.fontawesomeicons.solid.User
import harp.app.generated.resources.Res
import harp.app.generated.resources.logo
import io.github.vinceglb.filekit.dialogs.compose.rememberDirectoryPickerLauncher
import io.github.vinceglb.filekit.name
import kotlinx.coroutines.launch
import org.jetbrains.compose.resources.painterResource
import xyz.lepisma.harp.LocalRecentProfiles
import xyz.lepisma.harp.data.ProfileInfo
import xyz.lepisma.harp.data.ProfileType
import xyz.lepisma.harp.data.tags
import xyz.lepisma.harp.data.toastNotify
import xyz.lepisma.harp.screens.documents.DocumentsView
import xyz.lepisma.harp.screens.journals.JournalEntryForm
import xyz.lepisma.harp.screens.journals.JournalView
import xyz.lepisma.harp.screens.metrics.MetricsView
import xyz.lepisma.harp.screens.reminders.RemindersView
import xyz.lepisma.harp.viewmodel.ProfileViewModel


class ProfileScreen(val profileUuid: String) : Screen {
    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    override fun Content() {
        val navigator = LocalNavigator.currentOrThrow
        val profileViewModel = remember { ProfileViewModel() }
        val recentProfiles = LocalRecentProfiles.current

        LaunchedEffect(profileUuid) {
            val profileInfo = recentProfiles.all().find { it.uuid == profileUuid }
            if (profileInfo != null) {
                profileViewModel.loadProfile(profileInfo.profileFile, profileInfo.dataDir)
            }
        }

        val profile by profileViewModel.profile.collectAsState()
        val profileFile by profileViewModel.file.collectAsState()
        val dataDirPath by profileViewModel.dataDirPath.collectAsState()

        var selectedNavIndex by remember { mutableIntStateOf(0) }
        var showFormSheet by remember { mutableStateOf(false) }
        val sheetState = rememberModalBottomSheetState(skipPartiallyExpanded = true)

        var selectedJournalIdx by remember { mutableStateOf(0) }

        var showDataDirDialog by remember { mutableStateOf(false) }

        var showMetricsHelp by remember { mutableStateOf(false) }
        var showDocumentsHelp by remember { mutableStateOf(false) }

        val scope = rememberCoroutineScope()
        val uriHandler = LocalUriHandler.current

        if (showMetricsHelp) {
            AlertDialog(
                onDismissRequest = { showMetricsHelp = false },
                text = {
                    Column {
                        FlowRow {
                            Text("Ability to define new metrics is in development. In the meantime you can manually edit the backing Org mode file. See the ")

                            Text(
                                text = "documentation",
                                color = MaterialTheme.colorScheme.primary,
                                textDecoration = TextDecoration.Underline,
                                modifier = Modifier.clickable {
                                    uriHandler.openUri("https://docs.lepisma.xyz/harp/format/#metric-definitions")
                                }
                            )

                            Text(" for more details.")
                        }
                    }
                },
                confirmButton = {
                    TextButton(onClick = { showMetricsHelp = false }) {
                        Text("Dismiss")
                    }
                }
            )
        }

        if (showDocumentsHelp) {
            AlertDialog(
                onDismissRequest = { showDocumentsHelp = false },
                text = {
                    Column {
                        FlowRow {
                            Text("Ability to add new documents is in development. In the meantime you can manually edit the backing Org mode file. See the ")

                            Text(
                                text = "documentation",
                                color = MaterialTheme.colorScheme.primary,
                                textDecoration = TextDecoration.Underline,
                                modifier = Modifier.clickable {
                                    uriHandler.openUri("https://docs.lepisma.xyz/harp/format/#documents")
                                }
                            )

                            Text(" for more details.")
                        }
                    }
                },
                confirmButton = {
                    TextButton(onClick = { showDocumentsHelp = false }) {
                        Text("Dismiss")
                    }
                }
            )
        }

        profile?.let { profile ->
            Scaffold(
                floatingActionButton = {
                    when (selectedNavIndex) {
                        0 -> FloatingActionButton(onClick = { showFormSheet = true }) {
                            Icon(
                                imageVector = FontAwesomeIcons.Solid.Plus,
                                contentDescription = "Add Entry",
                                modifier = Modifier.size(20.dp)
                            )
                        }
                        1 -> FloatingActionButton(onClick = { showMetricsHelp = true }) {
                            Icon(
                                imageVector = FontAwesomeIcons.Solid.Plus,
                                contentDescription = "Define Metric",
                                modifier = Modifier.size(20.dp)
                            )
                        }
                        2 -> FloatingActionButton(onClick = { showDocumentsHelp = true }) {
                            Icon(
                                imageVector = FontAwesomeIcons.Solid.Plus,
                                contentDescription = "Add Document",
                                modifier = Modifier.size(20.dp)
                            )
                        }
                    }
                },
                bottomBar = {
                    NavigationBar {
                        listOf(
                            "Journals" to FontAwesomeIcons.Regular.StickyNote,
                            "Metrics" to FontAwesomeIcons.Regular.ChartBar,
                            "Documents" to FontAwesomeIcons.Solid.NotesMedical,
                            "Reminders" to FontAwesomeIcons.Regular.Bell
                        ).forEachIndexed { index, (label, icon) ->
                            NavigationBarItem(
                                selected = (selectedNavIndex == index),
                                onClick = { selectedNavIndex = index },
                                enabled = index == 0 || index == 1 || index == 2,
                                icon = {
                                    Icon(
                                        icon,
                                        modifier = Modifier.size(18.dp),
                                        contentDescription = label
                                    )
                                },
                                label = { Text(label) }
                            )
                        }
                    }
                }
            ) { innerPadding ->
                Column(
                    modifier = Modifier
                        .padding(innerPadding)
                ) {
                    Column(modifier = Modifier.padding(horizontal = 20.dp)) {
                        Row(verticalAlignment = Alignment.CenterVertically) {
                            Icon(
                                painter = painterResource(Res.drawable.logo),
                                contentDescription = "Logo",
                                modifier = Modifier.size(20.dp)
                            )

                            Text(
                                text = "harp",
                                style = MaterialTheme.typography.displaySmall,
                                fontStyle = FontStyle.Italic,
                                fontWeight = FontWeight.Bold,
                                color = MaterialTheme.colorScheme.outline,
                                modifier = Modifier
                                    .clickable {
                                        navigator.pop()
                                    }
                                    .padding(start = 5.dp)
                            )
                        }
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            modifier = Modifier.padding(vertical = 10.dp)
                        ) {
                            Icon(
                                when (profile.type) {
                                    ProfileType.CAT -> FontAwesomeIcons.Solid.Cat
                                    ProfileType.DOG -> FontAwesomeIcons.Solid.Dog
                                    else -> FontAwesomeIcons.Solid.User
                                },
                                contentDescription = "Profile Icon",
                                modifier = Modifier
                                    .size(24.dp),
                                tint = MaterialTheme.colorScheme.outline
                            )
                            Text(
                                text = profile.name,
                                style = MaterialTheme.typography.displaySmall,
                                modifier = Modifier.padding(start = 12.dp)
                            )
                        }

                        if (dataDirPath == null) {
                            Admonition(AdmonitionType.INFO) {
                                Column {
                                    Text("Data directory not set for this profile. Attachments will not work.")
                                    Button(
                                        onClick = {
                                            showDataDirDialog = true
                                        },
                                        contentPadding = PaddingValues(
                                            horizontal = 8.dp,
                                            vertical = 4.dp
                                        ),
                                        modifier = Modifier
                                            .height(28.dp)
                                            .padding(top = 4.dp)
                                    ) {
                                        Text("Set", style = MaterialTheme.typography.labelMedium)
                                    }
                                }
                            }
                            Spacer(Modifier.height(10.dp))

                            val dataDirPickerLauncher = rememberDirectoryPickerLauncher(
                                title = "Data directory picker"
                            ) { directory ->
                                if (directory == null) {
                                    toastNotify("Unable to pick data directory")
                                    return@rememberDirectoryPickerLauncher
                                }

                                if (directory.name != "data") {
                                    toastNotify("Unable to work with data directory named ${directory.name}")
                                    return@rememberDirectoryPickerLauncher
                                }

                                profileViewModel.setDataDirPath(directory)

                                scope.launch {
                                    recentProfiles.rememberProfile(
                                        ProfileInfo(
                                            uuid = profile.uuid,
                                            name = profile.name,
                                            type = profile.type,
                                            profileFile = profileFile!!,
                                            dataDir = directory
                                        )
                                    )
                                }
                            }

                            if (showDataDirDialog) {
                                AlertDialog(
                                    onDismissRequest = {
                                        showDataDirDialog = false
                                    },
                                    title = { Text("Data directory") },
                                    text = {
                                        Column {
                                            Text("Select the folder named 'data' for this profile in the same directory as the Org file. If it doesn’t exist, create it first.")
                                        }
                                    },
                                    confirmButton = {
                                        TextButton(onClick = {
                                            dataDirPickerLauncher.launch()
                                            showDataDirDialog = false
                                        }) {
                                            Text("Continue")
                                        }
                                    }
                                )
                            }
                        }
                    }

                    val profileTags = profile.tags()
                    if (profileTags.isNotEmpty()) {
                        TagsSelector(
                            tags = profileTags,
                            viewModel = profileViewModel
                        )
                    }

                    Column(modifier = Modifier.padding(horizontal = 20.dp)) {
                        when (selectedNavIndex) {
                            0 -> JournalView(
                                journals = profile.journals,
                                viewModel = profileViewModel,
                                onJournalIdxChange = { selectedJournalIdx = it }
                            )

                            1 -> MetricsView(
                                profile,
                                viewModel = profileViewModel
                            )

                            2 -> DocumentsView(
                                documents = profile.documents,
                                viewModel = profileViewModel,
                            )

                            3 -> RemindersView(
                                reminders = profile.reminders
                            )
                        }
                    }

                    if (showFormSheet) {
                        ModalBottomSheet(
                            onDismissRequest = { showFormSheet = false },
                            sheetState = sheetState
                        ) {
                            when (selectedNavIndex) {
                                0 -> JournalEntryForm(
                                    journalName = profile.journals[selectedJournalIdx].name,
                                    onSave = { entry ->
                                        profileViewModel.addJournalEntry(
                                            journalName = profile.journals[selectedJournalIdx].name,
                                            entry = entry
                                        )
                                        showFormSheet = false
                                    })
                            }
                        }
                    }
                }
            }
        }
    }
}