package xyz.lepisma.harp.data

import kotlinx.datetime.LocalDate
import kotlinx.datetime.LocalTime
import xyz.lepisma.orgmode.OrgInlineElem
import xyz.lepisma.orgmode.OrgPriority
import xyz.lepisma.orgmode.OrgSection

enum class RepetitionType {
    REGULAR,          // +  Usual org mode repetition
    FROM_NOW,         // .+ Add the repetition duration from the moment we mark it done
    SKIP_TILL_FUTURE, // ++ Move the repetition timer till it just hits future
}

enum class TimeUnit {
    HOUR,
    DAY,
    WEEK,
    MONTH,
    YEAR
}

// This should go in the parse library itself
data class Repeater(
    val type: RepetitionType,
    val value: Int,
    val unit: TimeUnit
)

data class WarningPeriod(
    val value: Int,
    val unit: TimeUnit
)

data class Datetime (
    val date: LocalDate,
    val time: Pair<LocalTime, LocalTime?>?,
    val isActive: Boolean,
    val repeater: Repeater?,
    val warningPeriod: WarningPeriod?
)

data class Reminder (
    val id: String,
    val title: String,
    val tags: List<String>,
    val body: String?,
    val scheduledStamp: Datetime?,
    val deadlineStamp: Datetime?,
    val stamp: Datetime?,
    val priority: OrgPriority?,
)

fun OrgInlineElem.DTStamp.toReminderDateTime(): Datetime? {
    val repeaterString = this.repeater
    val repeater = if (repeaterString != null) {
        val unit = when(repeaterString.last()) {
            'h' -> TimeUnit.HOUR
            'd' -> TimeUnit.DAY
            'w' -> TimeUnit.WEEK
            'm' -> TimeUnit.MONTH
            'y' -> TimeUnit.YEAR
            else -> return null
        }

        var value = 0

        val type = if (repeaterString.startsWith("++")) {
            value = repeaterString.substring(2, repeaterString.length - 1).toInt()
            RepetitionType.SKIP_TILL_FUTURE
        } else if (repeaterString.startsWith(".")) {
            value = repeaterString.substring(1, repeaterString.length - 1).toInt()
            RepetitionType.FROM_NOW
        } else if (repeaterString.startsWith("+")) {
            value = repeaterString.substring(1, repeaterString.length - 1).toInt()
            RepetitionType.REGULAR
        } else {
            return null
        }

        Repeater(type, value, unit)
    } else {
        null
    }

    return Datetime(
        date = this.date,
        time = this.time,
        isActive = this.isActive,
        repeater = repeater,
        // TODO: Parse this also
        warningPeriod = null
    )
}

fun parseReminder(section: OrgSection): Reminder? {
    if (section.heading.level.level != 2) {
        return null
    }

    val props = section.heading.properties?.map
    if (props == null) {
        return null
    }

    if (!props.contains("ID")) {
        return null
    }

    // There should be at least a scheduled or a deadline or plain stamp
    val scheduledElem = section.heading.planningInfo?.scheduled
    val deadlineElem = section.heading.planningInfo?.deadline
    if (scheduledElem == null && deadlineElem == null) {
        return null
    }

    val body = section.body.joinToString("") { it.tokens.joinToString("") { it.text } }

    return Reminder(
        id = orgLineToString(props["ID"]!!).trim(),
        title = orgLineToString(section.heading.title).trim(),
        // We don't parse body for tags in reminders
        tags = section.heading.tags?.tags ?: emptyList(),
        body = body,
        scheduledStamp = scheduledElem?.toReminderDateTime(),
        deadlineStamp = deadlineElem?.toReminderDateTime(),
        // TODO: Parse plain stamp too
        stamp = null,
        priority = section.heading.priority
    )
}