package xyz.lepisma.harp.data

import com.russhwolf.settings.Settings
import io.github.vinceglb.filekit.exists
import kotlinx.serialization.json.Json

private const val RECENT_PROFILE_PREFIX = "recent_profile_"

/**
 * Store for keeping recent profile information in platform independent way
 */
class RecentProfiles() {
    private val profilesInfo = mutableMapOf<String, ProfileInfo>()
    private val settings = Settings()

    init {
        settings.keys
            .filter { it.startsWith(RECENT_PROFILE_PREFIX) }
            .forEach { key ->
                val uuid = key.removePrefix(RECENT_PROFILE_PREFIX)
                val encodedString = settings.getStringOrNull(key)!!
                try {
                    val persistedProfileInfo = Json.decodeFromString<PersistedProfileInfo>(encodedString)
                    val profileInfo = persistedProfileInfo.toRuntime().getOrThrow()

                    // We only check that the main profile file exists, nothing else
                    if (profileInfo.profileFile.filePath.exists()) {
                        profilesInfo[uuid] = profileInfo
                    } else {
                        toastNotify("Profile file not present for id $uuid, forgetting it.")
                        forgetProfile(uuid)
                    }
                } catch (e: Exception) {
                    toastNotify("Unable to load profile info for id $uuid, forgetting it")
                    forgetProfile(uuid)
                }
            }
    }

    private fun key(uuid: String) = "$RECENT_PROFILE_PREFIX$uuid"

    /**
     * Remember profile information, overwriting any existing instance
     */
    suspend fun rememberProfile(profileInfo: ProfileInfo) {
        val uuid = profileInfo.uuid
        profilesInfo[uuid] = profileInfo

        // Also persist in settings
        val persistedProfileInfo = profileInfo.toPersistable()
        settings.putString(key(uuid), Json.encodeToString(persistedProfileInfo))
    }

    fun forgetProfile(uuid: String) {
        profilesInfo.remove(uuid)
        settings.remove(key(uuid))
    }

    fun all(): List<ProfileInfo> {
        return profilesInfo.values.toList()
    }
}