package xyz.lepisma.harp.data

import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.bookmarkData
import io.github.vinceglb.filekit.fromBookmarkData
import io.github.vinceglb.filekit.readString
import kotlinx.serialization.Serializable
import xyz.lepisma.orgmode.lexer.OrgLexer
import xyz.lepisma.orgmode.parse
import kotlin.io.encoding.Base64

/**
 * Minimal info of profile without actual data
 */
data class ProfileInfo(
    val uuid: String,
    val name: String,
    val type: ProfileType = ProfileType.HUMAN,
    val profileFile: FileUnderDir,
    val dataDir: PlatformFile?
)

/**
 * Read profile information using the files provided
 */
suspend fun readProfileInfo(profileFile: PlatformFile, dataDir: PlatformFile?): Result<ProfileInfo> {
    val text = profileFile.readString()
    
    try {
        val parsed = parse(OrgLexer(text).tokenize()) 
            ?: return Result.failure(Exception("Unable to parse profileFile Org file"))
        
        // We can skip fully parsing the document but that's an optimization for later
        val profile = fromOrgDocument(parsed).getOrThrow()
        
        return Result.success(ProfileInfo(
            uuid = profile.uuid,
            name = profile.name,
            type = profile.type,
            profileFile = FileUnderDir.Direct(profileFile),
            dataDir = dataDir
        ))
    } catch (e: Exception) {
        return Result.failure(e)
    }
}

/**
 * Serializable version of ProfileInfo
 */
@Serializable
data class PersistedProfileInfo(
    val uuid: String,
    val name: String,
    val type: ProfileType,
    val profileFileBookmarkData: PersistedFileUnderDir,
    val dataDirBookmarkData: String?
)

suspend fun ProfileInfo.toPersistable(): PersistedProfileInfo {
    return PersistedProfileInfo(
        uuid,
        name,
        type,
        profileFile.toPersistable(),
        if (dataDir != null) Base64.encode(dataDir.bookmarkData().bytes) else null
    )
}

fun PersistedProfileInfo.toRuntime(): Result<ProfileInfo> {
    return try {
        val fileUnderDir = profileFileBookmarkData.toRuntime().getOrThrow()
        Result.success(ProfileInfo(
            uuid,
            name,
            type,
            fileUnderDir,
            if (dataDirBookmarkData != null) PlatformFile.fromBookmarkData(Base64.decode(dataDirBookmarkData)) else null,
        ))
    } catch (e: Exception) {
        Result.failure(e)
    }
}