package xyz.lepisma.harp.data

import kotlinx.datetime.LocalDateTime
import xyz.lepisma.orgmode.OrgChunk
import xyz.lepisma.orgmode.OrgInlineElem
import xyz.lepisma.orgmode.OrgSection
import xyz.lepisma.orgmode.lexer.OrgLexer
import xyz.lepisma.orgmode.lexer.Token

data class Document(
    val name: String,
    val datetime: LocalDateTime,
    val uuid: String,
    val tags: List<String>,
    val source: Source?,
    val assets: List<Asset>,
    val metricValues: List<MetricValue>,
    val annotation: String?,
    val isPrivate: Boolean
)

fun parseDocumentEntry(section: OrgSection): Document? {
    // The format enforces a certain nesting structure
    if (section.heading.level.level != 2) {
        return null
    }

    val props = section.heading.properties?.map
    if (props == null) {
        return null
    }

    if (!props.contains("ID") || !props.contains("DATETIME") || !props.contains("FILES")) {
        return null
    }
    val id = orgLineToString(props["ID"]!!).trim()

    // HACK: Prop values are parsed as plain text for now so we will invoke the datetime parser manually
    // till the newer parsers handle the plain text well
    val dtTokens = OrgLexer(props["DATETIME"]?.items[0]?.tokens[0]?.text ?: "").tokenize()
    if (dtTokens.size != 3) {
        // Tokens will be SOF, DatetimeStamp, and EOF
        return null
    }
    val dtStamp = dtTokens[1] as Token.DatetimeStamp
    val dt = LocalDateTime(
        year = dtStamp.date.year,
        monthNumber = dtStamp.date.monthNumber,
        dayOfMonth = dtStamp.date.dayOfMonth,
        hour = dtStamp.time?.first?.hour ?: 0,
        minute = dtStamp.time?.first?.minute ?: 0,
        second = dtStamp.time?.first?.second ?: 0
    )

    val source: Source? = props.get("SOURCE")?.let { sourceLine ->
        Source(
            id = orgLineToString(sourceLine).trim(),
            description = "NA"
        )
    }

    val assets: List<Asset> = orgLineToString(props["FILES"]!!)
        .split(","
        ).map { fileName ->
            Asset(
                fileName.trim(),
                mimeType = null,
                text = null
            )
        }

    // TODO:
    // Parse assets list from tag, body, and attachment dir (not present now)
    // Parse PRIVATE from prop

    val body = section.body.joinToString("") { it.tokens.joinToString("") { it.text } }.trim()

    val metricValues = mutableListOf<MetricValue>()
    val tagSet = (section.heading.tags?.tags ?: emptyList()).toMutableSet()

    section.body.forEach { chunk ->
        when (chunk) {
            // TODO: We only parse plain paragraph for this right now, finish it
            is OrgChunk.OrgParagraph -> chunk.items.forEach { it ->
                when (it) {
                    is OrgInlineElem.HashMetric -> {
                        metricValues.add(
                            MetricValue(
                                id = it.metric,
                                datetime = dt,
                                value = it.value.toFloat(),
                                reference = id
                            )
                        )
                    }
                    is OrgInlineElem.HashTag -> {
                        tagSet.add(it.text)
                    }
                    else -> {  }
                }
            }
            else -> { }
        }
    }

    return Document(
        uuid = id,
        name = orgLineToString(section.heading.title).trim(),
        source = source,
        datetime = dt,
        tags = tagSet.toList().sorted(),
        metricValues = metricValues,
        annotation = body,
        assets = assets,
        isPrivate = false
    )
}