/*
 * Copyright (c)  2018 - 2021 Daniel Quah
 * Copyright (c)  2021 ASDF Dev Pte. Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package xyz.hisname.fireflyiii.workers.category

import android.content.Context
import androidx.work.*
import xyz.hisname.fireflyiii.data.local.dao.AppDatabase
import xyz.hisname.fireflyiii.data.local.pref.AppPref
import xyz.hisname.fireflyiii.data.remote.firefly.api.CategoryService
import xyz.hisname.fireflyiii.repository.category.CategoryRepository
import xyz.hisname.fireflyiii.util.network.HttpConstants
import xyz.hisname.fireflyiii.workers.BaseWorker
import xyz.hisname.fireflyiii.workers.DeleteCurrencyWorker
import java.time.Duration

class DeleteCategoryWorker(private val context: Context, workerParameters: WorkerParameters): BaseWorker(context, workerParameters) {

    companion object {
        fun initPeriodicWorker(categoryId: Long, context: Context, uuid: String) {
            val categoryTag =
                    WorkManager.getInstance(context).getWorkInfosByTag("delete_periodic_category_$categoryId" + "_$uuid").get()
            if (categoryTag == null || categoryTag.size == 0) {
                val categoryData = Data.Builder()
                    .putLong("categoryId", categoryId)
                    .putString("uuid", uuid)
                    .build()
                val appPref = AppPref(context.getSharedPreferences("$uuid-user-preferences", Context.MODE_PRIVATE))
                val delay = appPref.workManagerDelay
                val battery = appPref.workManagerLowBattery
                val networkType = appPref.workManagerNetworkType
                val requireCharging = appPref.workManagerRequireCharging
                val deleteCategoryWork = PeriodicWorkRequestBuilder<DeleteCurrencyWorker>(Duration.ofMinutes(delay))
                    .setInputData(categoryData)
                    .addTag("delete_periodic_category_$categoryId" + "_$uuid")
                    .addTag(uuid)
                    .setConstraints(Constraints.Builder()
                        .setRequiredNetworkType(networkType)
                        .setRequiresBatteryNotLow(battery)
                        .setRequiresCharging(requireCharging)
                        .build())
                    .build()
                WorkManager.getInstance(context).enqueue(deleteCategoryWork)
            }
        }

        fun cancelWorker(categoryId: Long, context: Context){
            WorkManager.getInstance(context).cancelAllWorkByTag("delete_periodic_category_$categoryId")
        }
    }

    override suspend fun doWork(): Result {
        val categoryDao = AppDatabase.getInstance(context, uuid).categoryDataDao()
        val categoryId = inputData.getLong("categoryId", 0)
        val repository = CategoryRepository(categoryDao, genericService(uuid).create(CategoryService::class.java))
        return when (repository.deleteCategoryById(categoryId)) {
            HttpConstants.NO_CONTENT_SUCCESS -> {
                cancelWorker(categoryId, context)
                Result.success()
            }
            HttpConstants.FAILED -> {
                Result.retry()
            }
            else -> {
                Result.failure()
            }
        }
    }
}