import { ContentGutters } from "@/src/components/gutters";
import { memo, useCallback, useEffect, useMemo, useRef, useState } from "react";
import { useCommunity, usePost, usePosts } from "@/src/lib/api";
import _ from "lodash";
import { IonContent, IonHeader, IonPage, IonToolbar } from "@ionic/react";
import { useUpdateRecentCommunity } from "@/src/stores/recent-communities";
import { BiHelpCircle } from "react-icons/bi";
import { UserDropdown } from "@/src/components/nav";
import { PageTitle } from "@/src/components/page-title";
import { useFiltersStore } from "@/src/stores/filters";
import {
  useHideTabBarOnMount,
  useIsActiveRoute,
  useKeyboardShortcut,
  useMedia,
  useNavbarHeight,
  useTabbarHeight,
  useUrlSearchState,
} from "@/src/lib/hooks";
import { ToolbarTitle } from "@/src/components/toolbar/toolbar-title";
import { getAccountSite, useAuth } from "@/src/stores/auth";
import { ToolbarBackButton } from "@/src/components/toolbar/toolbar-back-button";
import { cn } from "@/src/lib/utils";
import {
  PostCommentsButton,
  PostShareButton,
  PostVoting,
  usePostVoting,
} from "@/src/components/posts/post-buttons";
import { usePostsStore } from "@/src/stores/posts";
import z from "zod";
import { decodeApId, encodeApId } from "@/src/lib/api/utils";
import { useLinkContext } from "@/src/routing/link-context";
import { useParams } from "@/src/routing";
import { Forms } from "@/src/lib/api/adapters/api-blueprint";
import { ToolbarButtons } from "@/src/components/toolbar/toolbar-buttons";
import { Button } from "@/src/components/ui/button";
import { IonButtons, IonButton, IonModal, IonTitle } from "@ionic/react";
import { MarkdownRenderer } from "@/src/components/markdown/renderer";
import { Spinner, NoImage } from "@/src/components/icons";
import { getPostEmbed } from "@/src/lib/post";
import { useCommunityFromStore } from "@/src/stores/communities";
import { Swiper, SwiperRef, SwiperSlide } from "swiper/react";
import { Virtual, Mousewheel } from "swiper/modules";
import "swiper/css";
import "swiper/css/virtual";
import "swiper/css/mousewheel";
import { Swiper as SwiperType } from "swiper/types";
import { ProgressiveImage } from "@/src/components/progressive-image";
import { PanzoomProvider, usePanZoom } from "./panzoom";

const EMPTY_ARR: never[] = [];

function KeyboardShortcutHelpModal({
  isOpen,
  onClose,
}: {
  isOpen: boolean;
  onClose: () => void;
}) {
  return (
    <IonModal isOpen={isOpen} onDidDismiss={onClose}>
      <IonHeader>
        <IonToolbar>
          <IonButtons slot="start">
            <IonButton onClick={onClose}>Close</IonButton>
          </IonButtons>
          <IonTitle>Keyboard shortcuts</IonTitle>
        </IonToolbar>
      </IonHeader>
      <IonContent className="ion-padding">
        <MarkdownRenderer
          markdown={`
**Option 1: Arrow keys**

* Left arrow - previous post
* Right arrow - next post
* Up arrow - toggle upvote
* Down arrow - toggle downvote

**Option 2: For the gamers**

* A - previous post
* D - next post
* W - toggle upvote
* S - toggle downvote

**Option 3: Vim (god's editor)**

* j - previous post
* k - next post
* h - toggle upvote
* l - toggle downvote
`}
        />
      </IonContent>
    </IonModal>
  );
}

const Post = memo(
  ({
    apId,
    paddingTop,
    paddingBottom,
    onZoom,
    active,
  }: {
    apId: string;
    paddingTop: number;
    paddingBottom: number;
    onZoom: (scale: number) => void;
    active: boolean;
  }) => {
    const ref = useRef<HTMLDivElement>(null);

    const getCachePrefixer = useAuth((s) => s.getCachePrefixer);
    const postView = usePostsStore(
      (s) => s.posts[getCachePrefixer()(apId)]?.data,
    );

    const blurNsfw =
      useAuth((s) => getAccountSite(s.getSelectedAccount())?.blurNsfw) ?? true;
    const blurImg = blurNsfw ? postView?.nsfw : false;

    const embed = postView ? getPostEmbed(postView) : null;
    const img = embed?.thumbnail;

    const [ar, setAr] = useState(1);

    const zoom = usePanZoom(
      {
        container: ref.current,
        active,
        onZoom,
        imageAspectRatio: ar,
        paddingTop: paddingTop,
        paddingBottom: paddingBottom,
      },
      [embed?.fullResThumbnail, img],
    );

    return img ? (
      <div className="h-full w-full relative">
        <div
          ref={ref}
          className={cn("h-full w-full", zoom === 1 && "cursor-default!")}
        >
          <ProgressiveImage
            lowSrc={img}
            highSrc={embed?.fullResThumbnail}
            style={{ top: paddingTop, bottom: paddingBottom }}
            className={cn(
              "absolute inset-x-0 bg-transparent overflow-visible",
              blurImg && "blur-3xl",
            )}
            imgClassName="object-contain"
            onAspectRatio={(ratio) => setAr(ratio)}
          />
        </div>
      </div>
    ) : (
      <NoImage className="absolute top-1/2 left-1/2 h-40 w-40 -translate-1/2 text-white" />
    );
  },
);

function useLightboxPostFeedData({
  communityName,
  listingType,
  activePostApId,
}: {
  communityName?: string;
  listingType: Forms.GetPosts["type"];
  activePostApId?: string | null;
}) {
  const postsQuery = usePosts(
    communityName
      ? {
          communitySlug: communityName,
        }
      : {
          type: listingType,
        },
  );

  const posts = useMemo(
    () =>
      _.uniq(postsQuery.data?.pages.flatMap((p) => p.imagePosts) ?? EMPTY_ARR),
    [postsQuery.data],
  );

  const initPostApId = useRef(activePostApId).current ?? undefined;
  const missingPost = useMemo(() => {
    return !!initPostApId && !posts.includes(initPostApId);
  }, [posts, initPostApId]);

  const initPostQuery = usePost({
    ap_id: initPostApId,
    enabled: missingPost,
  });

  const [data, dataKey] = useMemo(() => {
    if (missingPost && initPostQuery.isPending) {
      return [[], Math.random()] as const;
    }

    if (missingPost) {
      return [
        [...(initPostApId ? [initPostApId] : []), ...posts],
        Math.random(),
      ] as const;
    }

    return [posts, Math.random()] as const;
  }, [missingPost, initPostApId, initPostQuery.isPending, posts]);

  return {
    data,
    dataKey,
    initPostQuery,
    postsQuery,
  };
}

export default function LightBoxPostFeed() {
  useHideTabBarOnMount();

  const linkCtx = useLinkContext();
  const { communityName: communityNameEncoded } = useParams(
    `${linkCtx.root}c/:communityName/lightbox`,
  );
  const communityName = useMemo(
    () =>
      communityNameEncoded
        ? decodeURIComponent(communityNameEncoded)
        : undefined,
    [communityNameEncoded],
  );

  const [encodedApId, setEncodedApId] = useUrlSearchState(
    "apId",
    "",
    z.string(),
  );
  const decodedApId = encodedApId ? decodeApId(encodedApId) : null;
  const initPostApId = useRef(decodedApId).current ?? undefined;

  const media = useMedia();
  const navbar = useNavbarHeight();
  const tabbar = useTabbarHeight();
  const isActive = useIsActiveRoute();

  const bottomBarHeight = media.md
    ? Math.max(navbar.height, tabbar.height + tabbar.inset)
    : tabbar.height + tabbar.inset;

  const listingType = useFiltersStore((s) => s.listingType);

  const getCachePrefixer = useAuth((s) => s.getCachePrefixer);

  const { data, dataKey, initPostQuery, postsQuery } = useLightboxPostFeedData({
    communityName,
    listingType,
    activePostApId: decodedApId,
  });

  const isPending = initPostQuery.isPending || postsQuery.isPending;

  useEffect(() => {
    if (initPostQuery.isError) {
      setEncodedApId("");
    }
  }, [initPostQuery.isError, setEncodedApId]);

  const activeIndex = Math.max(
    data.findIndex((apId) => apId === decodedApId),
    0,
  );

  const postApId = data[activeIndex];
  const post = usePostsStore((s) =>
    postApId ? s.posts[getCachePrefixer()(postApId)]?.data : null,
  );

  useCommunity({
    name: communityName,
  });
  const community = useCommunityFromStore(communityName);

  useUpdateRecentCommunity(community?.communityView);

  const voting = usePostVoting(postApId);
  const { vote, isUpvoted, isDownvoted } = voting ?? {};

  useKeyboardShortcut(
    useCallback(
      (e) => {
        if (post && !e.metaKey) {
          switch (e.key) {
            case "ArrowUp":
            case "w":
            case "h":
              e.preventDefault();
              e.stopPropagation();
              vote?.({
                score: isUpvoted ? 0 : 1,
                postApId: post.apId,
                postId: post.id,
              });
              break;
            case "ArrowDown":
            case "s":
            case "l":
              e.preventDefault();
              e.stopPropagation();
              vote?.({
                score: isDownvoted ? 0 : -1,
                postApId: post.apId,
                postId: post.id,
              });
              break;
            default:
              break;
          }
        }
      },
      [vote, isUpvoted, isDownvoted, post],
    ),
  );

  const swiperRef = useRef<SwiperType>(null);

  const onIndexChange = useCallback(
    (newIndex: number) => {
      const newApId = data[newIndex];
      if (newApId && !isPending) {
        setEncodedApId(encodeApId(newApId));
      }
    },
    [data, isPending, setEncodedApId],
  );

  const [keyboardHelpModal, setKeyboardHelpModal] = useState(false);
  const [hideNav, setHideNav] = useState(false);
  const onZoom = useCallback((scale: number) => {
    setHideNav(scale > 1);
    try {
      if (scale > 1) {
        swiperRef.current?.disable();
      } else {
        swiperRef.current?.enable();
      }
    } catch (err) {
      console.error(err);
    }
  }, []);

  useKeyboardShortcut(
    useCallback(
      (e) => {
        if (!isActive || e.metaKey) return;
        switch (e.key) {
          case "ArrowLeft":
          case "a":
          case "j":
            e.preventDefault();
            e.stopPropagation();
            swiperRef.current?.enable();
            swiperRef.current?.slidePrev(0);
            break;
          case "d":
          case "k":
          case "ArrowRight":
            e.preventDefault();
            e.stopPropagation();
            swiperRef.current?.enable();
            swiperRef.current?.slideNext(0);
            break;
          default:
            break;
        }
      },
      [isActive],
    ),
  );

  const ref = useRef<SwiperRef>(null);

  return (
    <IonPage className="dark">
      <PageTitle>Image</PageTitle>
      <IonHeader>
        <IonToolbar
          style={{
            "--ion-toolbar-background": hideNav
              ? "transparent"
              : "var(--shad-background)",
            "--ion-toolbar-border-color": "var(--shad-border)",
          }}
          className={cn("dark transition-opacity", hideNav && "opacity-0")}
        >
          <ToolbarButtons side="left">
            <ToolbarBackButton />
            <ToolbarTitle size="sm" numRightIcons={1}>
              {post?.title ?? "Loading..."}
            </ToolbarTitle>
          </ToolbarButtons>
          <ToolbarButtons side="right">
            <UserDropdown />
          </ToolbarButtons>
        </IonToolbar>
      </IonHeader>
      <IonContent
        fullscreen
        style={{
          "--ion-background-color": "black",
        }}
        scrollY={false}
        className="absolute inset-0"
      >
        <KeyboardShortcutHelpModal
          isOpen={keyboardHelpModal}
          onClose={() => setKeyboardHelpModal(false)}
        />

        <PanzoomProvider>
          <Swiper
            ref={ref}
            key={dataKey}
            mousewheel={{
              enabled: true,
              forceToAxis: true,
            }}
            keyboard={false}
            allowTouchMove={!hideNav && !media.md}
            onSwiper={(s) => (swiperRef.current = s)}
            initialSlide={activeIndex}
            onSlideChange={(s) => onIndexChange(s.activeIndex)}
            modules={[Virtual, Mousewheel]}
            virtual
            slidesPerView={1}
            className="h-full"
            onReachEnd={() => {
              if (postsQuery.hasNextPage && !postsQuery.isFetchingNextPage) {
                postsQuery.fetchNextPage();
              }
            }}
          >
            {data.map((item, i) => (
              <SwiperSlide
                key={i}
                virtualIndex={i}
                className="relative !h-auto"
              >
                <Post
                  key={item}
                  apId={item}
                  paddingTop={navbar.height + navbar.inset}
                  paddingBottom={bottomBarHeight}
                  active={i === activeIndex}
                  onZoom={onZoom}
                />
              </SwiperSlide>
            ))}
          </Swiper>
        </PanzoomProvider>

        {data.length === 0 && isPending && (
          <Spinner className="absolute top-1/2 left-1/2 text-4xl -translate-1/2 text-white animate-spin" />
        )}

        <div
          className={cn(
            "border-t-[.5px] z-10 absolute bottom-0 inset-x-0 dark transition-opacity",
            hideNav &&
              "opacity-0 hover:opacity-100 hover:bg-background/75 hover:backdrop-blur-xl",
            !isActive && "hidden",
          )}
          style={{
            // This is kinda weird, but I thought it looked
            // better if the bottom controls height mated the
            // toolbar height on desktop.
            height: bottomBarHeight,
            paddingBottom: tabbar.inset,
          }}
        >
          <ContentGutters className="h-full">
            <div className="flex flex-row items-center gap-3">
              {postApId && (
                <PostShareButton
                  postApId={postApId}
                  className="bg-transparent"
                />
              )}
              <div className="flex-1" />
              <Button
                variant="ghost"
                className={cn(
                  "text-muted-foreground max-md:hidden",
                  (initPostApId
                    ? postApId !== initPostApId
                    : activeIndex > 0) &&
                    "opacity-0 hover:opacity-100 focus:opacity-100",
                )}
                onClick={() => setKeyboardHelpModal(true)}
              >
                Keyboard shortcuts
                <BiHelpCircle />
              </Button>
              <div className="flex-1" />
              {postApId && (
                <PostCommentsButton
                  postApId={postApId}
                  className="bg-transparent"
                />
              )}
              {postApId && <PostVoting key={postApId} apId={postApId} />}
            </div>
          </ContentGutters>
        </div>
      </IonContent>
    </IonPage>
  );
}
