import { Store } from '@tanstack/store';
import { OptionalKeys } from './types.cjs';
export interface BatcherState<TValue> {
    /**
     * Number of batch executions that have been completed
     */
    executionCount: number;
    /**
     * Whether the batcher has no items to process (items array is empty)
     */
    isEmpty: boolean;
    /**
     * Whether the batcher is waiting for the timeout to trigger batch processing
     */
    isPending: boolean;
    /**
     * Array of items currently queued for batch processing
     */
    items: Array<TValue>;
    /**
     * Number of items currently in the batch queue
     */
    size: number;
    /**
     * Current processing status - 'idle' when not processing, 'pending' when waiting for timeout
     */
    status: 'idle' | 'pending';
    /**
     * Total number of items that have been processed across all batches
     */
    totalItemsProcessed: number;
}
/**
 * Options for configuring a Batcher instance
 */
export interface BatcherOptions<TValue> {
    /**
     * Custom function to determine if a batch should be processed
     * Return true to process the batch immediately
     */
    getShouldExecute?: (items: Array<TValue>, batcher: Batcher<TValue>) => boolean;
    /**
     * Initial state for the batcher
     */
    initialState?: Partial<BatcherState<TValue>>;
    /**
     * Maximum number of items in a batch
     * @default Infinity
     */
    maxSize?: number;
    /**
     * Callback fired after a batch is processed
     */
    onExecute?: (batch: Array<TValue>, batcher: Batcher<TValue>) => void;
    /**
     * Callback fired after items are added to the batcher
     */
    onItemsChange?: (batcher: Batcher<TValue>) => void;
    /**
     * Whether the batcher should start processing immediately
     * @default true
     */
    started?: boolean;
    /**
     * Maximum time in milliseconds to wait before processing a batch.
     * If the wait duration has elapsed, the batch will be processed.
     * If not provided, the batch will not be triggered by a timeout.
     * @default Infinity
     */
    wait?: number | ((batcher: Batcher<TValue>) => number);
}
type BatcherOptionsWithOptionalCallbacks<TValue> = OptionalKeys<Required<BatcherOptions<TValue>>, 'initialState' | 'onExecute' | 'onItemsChange'>;
/**
 * A class that collects items and processes them in batches.
 *
 * Batching is a technique for grouping multiple operations together to be processed as a single unit.
 *
 * The Batcher provides a flexible way to implement batching with configurable:
 * - Maximum batch size (number of items per batch)
 * - Time-based batching (process after X milliseconds)
 * - Custom batch processing logic via getShouldExecute
 * - Event callbacks for monitoring batch operations
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the batcher
 * - Use `onExecute` callback to react to batch execution and implement custom logic
 * - Use `onItemsChange` callback to react to items being added or removed from the batcher
 * - The state includes batch execution count, total items processed, items, and running status
 * - State can be accessed via `batcher.store.state` when using the class directly
 * - When using framework adapters (React/Solid), state is accessed from `batcher.state`
 *
 * @example
 * ```ts
 * const batcher = new Batcher<number>(
 *   (items) => console.log('Processing batch:', items),
 *   {
 *     maxSize: 5,
 *     wait: 2000,
 *     onExecute: (batch, batcher) => console.log('Batch executed:', batch)
 *   }
 * );
 *
 * batcher.addItem(1);
 * batcher.addItem(2);
 * // After 2 seconds or when 5 items are added, whichever comes first,
 * // the batch will be processed
 * // batcher.flush() // manually trigger a batch
 * ```
 */
export declare class Batcher<TValue> {
    #private;
    fn: (items: Array<TValue>) => void;
    readonly store: Store<Readonly<BatcherState<TValue>>>;
    options: BatcherOptionsWithOptionalCallbacks<TValue>;
    constructor(fn: (items: Array<TValue>) => void, initialOptions: BatcherOptions<TValue>);
    /**
     * Updates the batcher options
     */
    setOptions: (newOptions: Partial<BatcherOptions<TValue>>) => void;
    /**
     * Adds an item to the batcher
     * If the batch size is reached, timeout occurs, or shouldProcess returns true, the batch will be processed
     */
    addItem: (item: TValue) => void;
    /**
     * Processes the current batch of items immediately
     */
    flush: () => void;
    /**
     * Returns a copy of all items in the batcher
     */
    peekAllItems: () => Array<TValue>;
    /**
     * Removes all items from the batcher
     */
    clear: () => void;
    /**
     * Resets the batcher state to its default values
     */
    reset: () => void;
}
/**
 * Creates a batcher that processes items in batches
 *
 * @example
 * ```ts
 * const batchItems = batch<number>(
 *   (items) => console.log('Processing:', items),
 *   {
 *     maxSize: 3,
 *     onExecute: (batch, batcher) => console.log('Batch executed:', batch)
 *   }
 * );
 *
 * batchItems(1);
 * batchItems(2);
 * batchItems(3); // Triggers batch processing
 * ```
 */
export declare function batch<TValue>(fn: (items: Array<TValue>) => void, options: BatcherOptions<TValue>): (item: TValue) => void;
export {};
