import { UTF_16 } from '../../utils/UTF_16.js';
import { arrayBufferToString } from '../../utils/arrayBufferToString.js';
import { base64decode } from '../../utils/base64decode.js';
import { getElementsByName } from '../../xml/getElementsByName.js';
import { parseXml } from '../../xml/parseXml.js';
import { CHALLENGE } from '../common/CHALLENGE.js';
import { PLAYREADY_KEY_MESSAGE } from '../common/PLAYREADY_KEY_MESSAGE.js';
/**
 * Gets the PlayReady license request from the MediaKeyMessageEvent.
 *
 * @param message - An ArrayBuffer from the content decryption module.
 * @param encoding - The message encoding type. Default is UTF-16.
 * @returns The license request as an ArrayBuffer.
 *
 * @group DRM
 * @beta
 *
 * @example
 * {@includeCode ../../../test/drm/playready/getLicenseRequestFromMessage.test.ts#example}
 */
export function getLicenseRequestFromMessage(message, encoding = UTF_16) {
    var _a;
    // If encoding is configured for UTF-16 and the number of bytes is odd,
    // assume an 'unwrapped' raw CDM message.
    if (encoding === UTF_16 && (message === null || message === void 0 ? void 0 : message.byteLength) % 2 === 1) {
        return message;
    }
    const msg = arrayBufferToString(message, encoding);
    const xml = parseXml(msg);
    const playReadyKeyMessage = getElementsByName(xml, PLAYREADY_KEY_MESSAGE)[0];
    if (!playReadyKeyMessage) {
        // The message from the CDM is not wrapped and contains the direct challenge.
        return message;
    }
    const challengeNode = getElementsByName(playReadyKeyMessage, CHALLENGE)[0];
    const challengeValue = (_a = challengeNode === null || challengeNode === void 0 ? void 0 : challengeNode.childNodes[0]) === null || _a === void 0 ? void 0 : _a.nodeValue;
    if (challengeValue) {
        return base64decode(challengeValue).buffer;
    }
    return message;
}
//# sourceMappingURL=getLicenseRequestFromMessage.js.map