"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLicenseRequestFromMessage = getLicenseRequestFromMessage;
const UTF_16_js_1 = require("../../utils/UTF_16.js");
const arrayBufferToString_js_1 = require("../../utils/arrayBufferToString.js");
const base64decode_js_1 = require("../../utils/base64decode.js");
const getElementsByName_js_1 = require("../../xml/getElementsByName.js");
const parseXml_js_1 = require("../../xml/parseXml.js");
const CHALLENGE_js_1 = require("../common/CHALLENGE.js");
const PLAYREADY_KEY_MESSAGE_js_1 = require("../common/PLAYREADY_KEY_MESSAGE.js");
/**
 * Gets the PlayReady license request from the MediaKeyMessageEvent.
 *
 * @param message - An ArrayBuffer from the content decryption module.
 * @param encoding - The message encoding type. Default is UTF-16.
 * @returns The license request as an ArrayBuffer.
 *
 * @group DRM
 * @beta
 *
 * @example
 * {@includeCode ../../../test/drm/playready/getLicenseRequestFromMessage.test.ts#example}
 */
function getLicenseRequestFromMessage(message, encoding = UTF_16_js_1.UTF_16) {
    var _a;
    // If encoding is configured for UTF-16 and the number of bytes is odd,
    // assume an 'unwrapped' raw CDM message.
    if (encoding === UTF_16_js_1.UTF_16 && (message === null || message === void 0 ? void 0 : message.byteLength) % 2 === 1) {
        return message;
    }
    const msg = (0, arrayBufferToString_js_1.arrayBufferToString)(message, encoding);
    const xml = (0, parseXml_js_1.parseXml)(msg);
    const playReadyKeyMessage = (0, getElementsByName_js_1.getElementsByName)(xml, PLAYREADY_KEY_MESSAGE_js_1.PLAYREADY_KEY_MESSAGE)[0];
    if (!playReadyKeyMessage) {
        // The message from the CDM is not wrapped and contains the direct challenge.
        return message;
    }
    const challengeNode = (0, getElementsByName_js_1.getElementsByName)(playReadyKeyMessage, CHALLENGE_js_1.CHALLENGE)[0];
    const challengeValue = (_a = challengeNode === null || challengeNode === void 0 ? void 0 : challengeNode.childNodes[0]) === null || _a === void 0 ? void 0 : _a.nodeValue;
    if (challengeValue) {
        return (0, base64decode_js_1.base64decode)(challengeValue).buffer;
    }
    return message;
}
//# sourceMappingURL=getLicenseRequestFromMessage.js.map