import { Store } from '@tanstack/store';
import { AnyFunction } from './types.js';
export interface RateLimiterState {
    /**
     * Number of function executions that have been completed
     */
    executionCount: number;
    /**
     * Array of timestamps when executions occurred for rate limiting calculations
     */
    executionTimes: Array<number>;
    /**
     * Whether the rate limiter has exceeded the limit
     */
    isExceeded: boolean;
    /**
     * Number of function executions that have been rejected due to rate limiting
     */
    rejectionCount: number;
    /**
     * Current execution status - 'disabled' when not active, 'executing' when executing, 'idle' when not executing, 'exceeded' when rate limit is exceeded
     */
    status: 'disabled' | 'exceeded' | 'idle';
}
/**
 * Options for configuring a rate-limited function
 */
export interface RateLimiterOptions<TFn extends AnyFunction> {
    /**
     * Whether the rate limiter is enabled. When disabled, maybeExecute will not trigger any executions.
     * Defaults to true.
     */
    enabled?: boolean | ((rateLimiter: RateLimiter<TFn>) => boolean);
    /**
     * Initial state for the rate limiter
     */
    initialState?: Partial<RateLimiterState>;
    /**
     * Maximum number of executions allowed within the time window.
     * Can be a number or a callback function that receives the rate limiter instance and returns a number.
     */
    limit: number | ((rateLimiter: RateLimiter<TFn>) => number);
    /**
     * Callback function that is called after the function is executed
     */
    onExecute?: (args: Parameters<TFn>, rateLimiter: RateLimiter<TFn>) => void;
    /**
     * Optional callback function that is called when an execution is rejected due to rate limiting
     */
    onReject?: (rateLimiter: RateLimiter<TFn>) => void;
    /**
     * Time window in milliseconds within which the limit applies.
     * Can be a number or a callback function that receives the rate limiter instance and returns a number.
     */
    window: number | ((rateLimiter: RateLimiter<TFn>) => number);
    /**
     * Type of window to use for rate limiting
     * - 'fixed': Uses a fixed window that resets after the window period
     * - 'sliding': Uses a sliding window that allows executions as old ones expire
     * Defaults to 'fixed'
     */
    windowType?: 'fixed' | 'sliding';
}
/**
 * A class that creates a rate-limited function.
 *
 * Rate limiting is a simple approach that allows a function to execute up to a limit within a time window,
 * then blocks all subsequent calls until the window passes. This can lead to "bursty" behavior where
 * all executions happen immediately, followed by a complete block.
 *
 * The rate limiter supports two types of windows:
 * - 'fixed': A strict window that resets after the window period. All executions within the window count
 *   towards the limit, and the window resets completely after the period.
 * - 'sliding': A rolling window that allows executions as old ones expire. This provides a more
 *   consistent rate of execution over time.
 *
 * For smoother execution patterns, consider using:
 * - Throttling: Ensures consistent spacing between executions (e.g. max once per 200ms)
 * - Debouncing: Waits for a pause in calls before executing (e.g. after 500ms of no calls)
 *
 * Rate limiting is best used for hard API limits or resource constraints. For UI updates or
 * smoothing out frequent events, throttling or debouncing usually provide better user experience.
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the rate limiter
 * - Use `onExecute` callback to react to function execution and implement custom logic
 * - Use `onReject` callback to react to executions being rejected when rate limit is exceeded
 * - The state includes execution count, execution times, and rejection count
 * - State can be accessed via `rateLimiter.store.state` when using the class directly
 * - When using framework adapters (React/Solid), state is accessed from `rateLimiter.state`
 *
 * @example
 * ```ts
 * const rateLimiter = new RateLimiter(
 *   (id: string) => api.getData(id),
 *   {
 *     limit: 5,
 *     window: 1000,
 *     windowType: 'sliding',
 *   }
 * );
 *
 * // Will execute immediately until limit reached, then block
 * rateLimiter.maybeExecute('123');
 * ```
 */
export declare class RateLimiter<TFn extends AnyFunction> {
    #private;
    fn: TFn;
    readonly store: Store<Readonly<RateLimiterState>>;
    options: RateLimiterOptions<TFn>;
    constructor(fn: TFn, initialOptions: RateLimiterOptions<TFn>);
    /**
     * Updates the rate limiter options
     */
    setOptions: (newOptions: Partial<RateLimiterOptions<TFn>>) => void;
    /**
     * Attempts to execute the rate-limited function if within the configured limits.
     * Will reject execution if the number of calls in the current window exceeds the limit.
     *
     * @example
     * ```ts
     * const rateLimiter = new RateLimiter(fn, { limit: 5, window: 1000 });
     *
     * // First 5 calls will return true
     * rateLimiter.maybeExecute('arg1', 'arg2'); // true
     *
     * // Additional calls within the window will return false
     * rateLimiter.maybeExecute('arg1', 'arg2'); // false
     * ```
     */
    maybeExecute: (...args: Parameters<TFn>) => boolean;
    /**
     * Returns the number of remaining executions allowed in the current window
     */
    getRemainingInWindow: () => number;
    /**
     * Returns the number of milliseconds until the next execution will be possible
     */
    getMsUntilNextWindow: () => number;
    /**
     * Resets the rate limiter state
     */
    reset: () => void;
}
/**
 * Creates a rate-limited function that will execute the provided function up to a maximum number of times within a time window.
 *
 * Note that rate limiting is a simpler form of execution control compared to throttling or debouncing:
 * - A rate limiter will allow all executions until the limit is reached, then block all subsequent calls until the window resets
 * - A throttler ensures even spacing between executions, which can be better for consistent performance
 * - A debouncer collapses multiple calls into one, which is better for handling bursts of events
 *
 * The rate limiter supports two types of windows:
 * - 'fixed': A strict window that resets after the window period. All executions within the window count
 *   towards the limit, and the window resets completely after the period.
 * - 'sliding': A rolling window that allows executions as old ones expire. This provides a more
 *   consistent rate of execution over time.
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the rate limiter
 * - Use `onExecute` callback to react to function execution and implement custom logic
 * - Use `onReject` callback to react to executions being rejected when rate limit is exceeded
 * - The state includes execution count, execution times, and rejection count
 * - State can be accessed via the underlying RateLimiter instance's `store.state` property
 * - When using framework adapters (React/Solid), state is accessed from the hook's state property
 *
 * Consider using throttle() or debounce() if you need more intelligent execution control. Use rate limiting when you specifically
 * need to enforce a hard limit on the number of executions within a time period.
 *
 * @example
 * ```ts
 * // Rate limit to 5 calls per minute with a sliding window
 * const rateLimited = rateLimit(makeApiCall, {
 *   limit: 5,
 *   window: 60000,
 *   windowType: 'sliding',
 *   onReject: (rateLimiter) => {
 *     console.log(`Rate limit exceeded. Try again in ${rateLimiter.getMsUntilNextWindow()}ms`);
 *   }
 * });
 *
 * // First 5 calls will execute immediately
 * // Additional calls will be rejected until the minute window resets
 * rateLimited();
 *
 * // For more even execution, consider using throttle instead:
 * const throttled = throttle(makeApiCall, { wait: 12000 }); // One call every 12 seconds
 * ```
 */
export declare function rateLimit<TFn extends AnyFunction>(fn: TFn, initialOptions: RateLimiterOptions<TFn>): (...args: Parameters<TFn>) => boolean;
