import { Store } from '@tanstack/store';
export interface QueuerState<TValue> {
    /**
     * Number of items that have been processed by the queuer
     */
    executionCount: number;
    /**
     * Number of items that have been removed from the queue due to expiration
     */
    expirationCount: number;
    /**
     * Whether the queuer has no items to process (items array is empty)
     */
    isEmpty: boolean;
    /**
     * Whether the queuer has reached its maximum capacity
     */
    isFull: boolean;
    /**
     * Whether the queuer is not currently processing any items
     */
    isIdle: boolean;
    /**
     * Whether the queuer is active and will process items automatically
     */
    isRunning: boolean;
    /**
     * Array of items currently waiting to be processed
     */
    items: Array<TValue>;
    /**
     * Timestamps when items were added to the queue for expiration tracking
     */
    itemTimestamps: Array<number>;
    /**
     * Whether the queuer has a pending timeout for processing the next item
     */
    pendingTick: boolean;
    /**
     * Number of items that have been rejected from being added to the queue
     */
    rejectionCount: number;
    /**
     * Number of items currently in the queue
     */
    size: number;
    /**
     * Current processing status - 'idle' when not processing, 'running' when active, 'stopped' when paused
     */
    status: 'idle' | 'running' | 'stopped';
}
/**
 * Options for configuring a Queuer instance.
 *
 * These options control queue behavior, item expiration, callbacks, and more.
 */
export interface QueuerOptions<TValue> {
    /**
     * Default position to add items to the queuer
     * @default 'back'
     */
    addItemsTo?: QueuePosition;
    /**
     * Maximum time in milliseconds that an item can stay in the queue
     * If not provided, items will never expire
     */
    expirationDuration?: number;
    /**
     * Function to determine if an item has expired
     * If provided, this overrides the expirationDuration behavior
     */
    getIsExpired?: (item: TValue, addedAt: number) => boolean;
    /**
     * Default position to get items from during processing
     * @default 'front'
     */
    getItemsFrom?: QueuePosition;
    /**
     * Function to determine priority of items in the queuer
     * Higher priority items will be processed first
     */
    getPriority?: (item: TValue) => number;
    /**
     * Initial items to populate the queuer with
     */
    initialItems?: Array<TValue>;
    /**
     * Initial state for the queuer
     */
    initialState?: Partial<QueuerState<TValue>>;
    /**
     * Maximum number of items allowed in the queuer
     */
    maxSize?: number;
    /**
     * Callback fired whenever an item is removed from the queuer
     */
    onExecute?: (item: TValue, queuer: Queuer<TValue>) => void;
    /**
     * Callback fired whenever an item expires in the queuer
     */
    onExpire?: (item: TValue, queuer: Queuer<TValue>) => void;
    /**
     * Callback fired whenever an item is added or removed from the queuer
     */
    onItemsChange?: (queuer: Queuer<TValue>) => void;
    /**
     * Callback fired whenever an item is rejected from being added to the queuer
     */
    onReject?: (item: TValue, queuer: Queuer<TValue>) => void;
    /**
     * Whether the queuer should start processing tasks immediately
     */
    started?: boolean;
    /**
     * Time in milliseconds to wait between processing items.
     * Can be a number or a function that returns a number.
     * @default 0
     */
    wait?: number | ((queuer: Queuer<TValue>) => number);
}
/**
 * Position type for addItem and getNextItem operations.
 *
 * - 'front': Operate on the front of the queue (FIFO)
 * - 'back': Operate on the back of the queue (LIFO)
 */
export type QueuePosition = 'front' | 'back';
/**
 * A flexible queue that processes items with configurable wait times, expiration, and priority.
 *
 * Features:
 * - Automatic or manual processing of items
 * - FIFO (First In First Out), LIFO (Last In First Out), or double-ended queue behavior
 * - Priority-based ordering when getPriority is provided
 * - Item expiration and removal of stale items
 * - Callbacks for queue state changes, execution, rejection, and expiration
 *
 * Running behavior:
 * - `start()`: Begins automatically processing items in the queue (defaults to isRunning)
 * - `stop()`: Pauses processing but maintains queue state
 * - `wait`: Configurable delay between processing items
 * - `onItemsChange`/`onExecute`: Callbacks for monitoring queue state
 *
 * Manual processing is also supported when automatic processing is disabled:
 * - `execute()`: Processes the next item using the provided function
 * - `getNextItem()`: Removes and returns the next item without processing
 *
 * Queue behavior defaults to FIFO:
 * - `addItem(item)`: Adds to the back of the queue
 * - Items processed from the front of the queue
 *
 * Priority queue:
 * - Provide a `getPriority` function; higher values are processed first
 *
 * Stack (LIFO):
 * - `addItem(item, 'back')`: Adds to the back
 * - `getNextItem('back')`: Removes from the back
 *
 * Double-ended queue:
 * - `addItem(item, position)`: Adds to specified position ('front'/'back')
 * - `getNextItem(position)`: Removes from specified position
 *
 * Item expiration:
 * - `expirationDuration`: Maximum time items can stay in the queue
 * - `getIsExpired`: Function to override default expiration
 * - `onExpire`: Callback for expired items
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the queuer
 * - Use `onExecute` callback to react to item execution and implement custom logic
 * - Use `onItemsChange` callback to react to items being added or removed from the queue
 * - Use `onExpire` callback to react to items expiring and implement custom logic
 * - Use `onReject` callback to react to items being rejected when the queue is full
 * - The state includes execution count, expiration count, rejection count, and isRunning status
 * - State can be accessed via `queuer.store.state` when using the class directly
 * - When using framework adapters (React/Solid), state is accessed from `queuer.state`
 *
 * Example usage:
 * ```ts
 * // Auto-processing queue with wait time
 * const autoQueue = new Queuer<number>((n) => console.log(n), {
 *   started: true, // Begin processing immediately
 *   wait: 1000, // Wait 1s between items
 *   onExecute: (item, queuer) => console.log(`Processed ${item}`)
 * });
 * autoQueue.addItem(1); // Will process after 1s
 * autoQueue.addItem(2); // Will process 1s after first item
 *
 * // Manual processing queue
 * const manualQueue = new Queuer<number>((n) => console.log(n), {
 *   started: false
 * });
 * manualQueue.addItem(1); // [1]
 * manualQueue.addItem(2); // [1, 2]
 * manualQueue.execute(); // logs 1, queue is [2]
 * manualQueue.getNextItem(); // returns 2, queue is empty
 * ```
 */
export declare class Queuer<TValue> {
    #private;
    fn: (item: TValue) => void;
    readonly store: Store<Readonly<QueuerState<TValue>>>;
    options: QueuerOptions<TValue>;
    constructor(fn: (item: TValue) => void, initialOptions?: QueuerOptions<TValue>);
    /**
     * Updates the queuer options. New options are merged with existing options.
     */
    setOptions: (newOptions: Partial<QueuerOptions<TValue>>) => void;
    /**
     * Adds an item to the queue. If the queue is full, the item is rejected and onReject is called.
     * Items can be inserted based on priority or at the front/back depending on configuration.
     *
     * Returns true if the item was added, false if the queue is full.
     *
     * Example usage:
     * ```ts
     * queuer.addItem('task');
     * queuer.addItem('task2', 'front');
     * ```
     */
    addItem: (item: TValue, position?: QueuePosition, runOnItemsChange?: boolean) => boolean;
    /**
     * Removes and returns the next item from the queue without executing the function.
     * Use for manual queue management. Normally, use execute() to process items.
     *
     * Example usage:
     * ```ts
     * // FIFO
     * queuer.getNextItem();
     * // LIFO
     * queuer.getNextItem('back');
     * ```
     */
    getNextItem: (position?: QueuePosition) => TValue | undefined;
    /**
     * Removes and returns the next item from the queue and processes it using the provided function.
     *
     * Example usage:
     * ```ts
     * queuer.execute();
     * // LIFO
     * queuer.execute('back');
     * ```
     */
    execute: (position?: QueuePosition) => TValue | undefined;
    /**
     * Processes a specified number of items to execute immediately with no wait time
     * If no numberOfItems is provided, all items will be processed
     */
    flush: (numberOfItems?: number, position?: QueuePosition) => void;
    /**
     * Processes all items in the queue as a batch using the provided function as an argument
     * The queue is cleared after processing
     */
    flushAsBatch: (batchFunction: (items: Array<TValue>) => void) => void;
    /**
     * Returns the next item in the queue without removing it.
     *
     * Example usage:
     * ```ts
     * queuer.peekNextItem(); // front
     * queuer.peekNextItem('back'); // back
     * ```
     */
    peekNextItem: (position?: QueuePosition) => TValue | undefined;
    /**
     * Returns a copy of all items in the queue.
     */
    peekAllItems: () => Array<TValue>;
    /**
     * Starts processing items in the queue. If already isRunning, does nothing.
     */
    start: () => void;
    /**
     * Stops processing items in the queue. Does not clear the queue.
     */
    stop: () => void;
    /**
     * Removes all pending items from the queue. Does not affect items being processed.
     */
    clear: () => void;
    /**
     * Resets the queuer state to its default values
     */
    reset: () => void;
}
/**
 * Creates a queue that processes items immediately upon addition.
 * Items are processed sequentially in FIFO order by default.
 *
 * This is a simplified wrapper around the Queuer class that only exposes the
 * `addItem` method. The queue is always isRunning and will process items as they are added.
 * For more control over queue processing, use the Queuer class directly.
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the queuer
 * - Use `onExecute` callback to react to item execution and implement custom logic
 * - Use `onItemsChange` callback to react to items being added or removed from the queue
 * - Use `onExpire` callback to react to items expiring and implement custom logic
 * - Use `onReject` callback to react to items being rejected when the queue is full
 * - The state includes execution count, expiration count, rejection count, and isRunning status
 * - State can be accessed via the underlying Queuer instance's `store.state` property
 * - When using framework adapters (React/Solid), state is accessed from the hook's state property
 *
 * Example usage:
 * ```ts
 * // Basic sequential processing
 * const processItems = queue<number>((n) => console.log(n), {
 *   wait: 1000,
 *   onItemsChange: (queuer) => console.log(queuer.peekAllItems())
 * });
 * processItems(1); // Logs: 1
 * processItems(2); // Logs: 2 after 1 completes
 *
 * // Priority queue
 * const processPriority = queue<number>((n) => console.log(n), {
 *   getPriority: n => n // Higher numbers processed first
 * });
 * processPriority(1);
 * processPriority(3); // Processed before 1
 * ```
 */
export declare function queue<TValue>(fn: (item: TValue) => void, initialOptions: QueuerOptions<TValue>): (item: TValue, position?: QueuePosition, runOnItemsChange?: boolean) => boolean;
