import { Store } from '@tanstack/store';
import { AnyFunction } from './types.cjs';
export interface DebouncerState<TFn extends AnyFunction> {
    /**
     * Whether the debouncer can execute on the leading edge of the timeout
     */
    canLeadingExecute: boolean;
    /**
     * Number of function executions that have been completed
     */
    executionCount: number;
    /**
     * Whether the debouncer is waiting for the timeout to trigger execution
     */
    isPending: boolean;
    /**
     * The arguments from the most recent call to maybeExecute
     */
    lastArgs: Parameters<TFn> | undefined;
    /**
     * Current execution status - 'idle' when not active, 'pending' when waiting for timeout
     */
    status: 'disabled' | 'idle' | 'pending';
}
/**
 * Options for configuring a debounced function
 */
export interface DebouncerOptions<TFn extends AnyFunction> {
    /**
     * Whether the debouncer is enabled. When disabled, maybeExecute will not trigger any executions.
     * Can be a boolean or a function that returns a boolean.
     * Defaults to true.
     */
    enabled?: boolean | ((debouncer: Debouncer<TFn>) => boolean);
    /**
     * Initial state for the debouncer
     */
    initialState?: Partial<DebouncerState<TFn>>;
    /**
     * Whether to execute on the leading edge of the timeout.
     * The first call will execute immediately and the rest will wait the delay.
     * Defaults to false.
     */
    leading?: boolean;
    /**
     * Callback function that is called after the function is executed
     */
    onExecute?: (args: Parameters<TFn>, debouncer: Debouncer<TFn>) => void;
    /**
     * Whether to execute on the trailing edge of the timeout.
     * Defaults to true.
     */
    trailing?: boolean;
    /**
     * Delay in milliseconds before executing the function.
     * Can be a number or a function that returns a number.
     * Defaults to 0ms
     */
    wait: number | ((debouncer: Debouncer<TFn>) => number);
}
/**
 * A class that creates a debounced function.
 *
 * Debouncing ensures that a function is only executed after a certain amount of time has passed
 * since its last invocation. This is useful for handling frequent events like window resizing,
 * scroll events, or input changes where you want to limit the rate of execution.
 *
 * The debounced function can be configured to execute either at the start of the delay period
 * (leading edge) or at the end (trailing edge, default). Each new call during the wait period
 * will reset the timer.
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the debouncer
 * - Use `onExecute` callback to react to function execution and implement custom logic
 * - The state includes canLeadingExecute, execution count, and isPending status
 * - State can be accessed via `debouncer.store.state` when using the class directly
 * - When using framework adapters (React/Solid), state is accessed from `debouncer.state`
 *
 * @example
 * ```ts
 * const debouncer = new Debouncer((value: string) => {
 *   saveToDatabase(value);
 * }, { wait: 500 });
 *
 * // Will only save after 500ms of no new input
 * inputElement.addEventListener('input', () => {
 *   debouncer.maybeExecute(inputElement.value);
 * });
 * ```
 */
export declare class Debouncer<TFn extends AnyFunction> {
    #private;
    fn: TFn;
    readonly store: Store<Readonly<DebouncerState<TFn>>>;
    options: DebouncerOptions<TFn>;
    constructor(fn: TFn, initialOptions: DebouncerOptions<TFn>);
    /**
     * Updates the debouncer options
     */
    setOptions: (newOptions: Partial<DebouncerOptions<TFn>>) => void;
    /**
     * Attempts to execute the debounced function
     * If a call is already in progress, it will be queued
     */
    maybeExecute: (...args: Parameters<TFn>) => void;
    /**
     * Processes the current pending execution immediately
     */
    flush: () => void;
    /**
     * Cancels any pending execution
     */
    cancel: () => void;
    /**
     * Resets the debouncer state to its default values
     */
    reset: () => void;
}
/**
 * Creates a debounced function that delays invoking the provided function until after a specified wait time.
 * Multiple calls during the wait period will cancel previous pending invocations and reset the timer.
 *
 * This the the simple function wrapper implementation pulled from the Debouncer class. If you need
 * more control over the debouncing behavior, use the Debouncer class directly.
 *
 * If leading option is true, the function will execute immediately on the first call, then wait the delay
 * before allowing another execution.
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the debouncer
 * - Use `onExecute` callback to react to function execution and implement custom logic
 * - The state includes canLeadingExecute, execution count, and isPending status
 * - State can be accessed via the underlying Debouncer instance's `store.state` property
 * - When using framework adapters (React/Solid), state is accessed from the hook's state property
 *
 * @example
 * ```ts
 * const debounced = debounce(() => {
 *   saveChanges();
 * }, { wait: 1000 });
 *
 * // Called repeatedly but executes at most once per second
 * inputElement.addEventListener('input', debounced);
 * ```
 */
export declare function debounce<TFn extends AnyFunction>(fn: TFn, initialOptions: DebouncerOptions<TFn>): (...args: Parameters<TFn>) => void;
