declare module "lots-o-nekos" {
  /**
   * An Oneko that the constructor method successfully initialized using the provided parameters.
   * 
   * ```ts
   * let cat = new Oneko();
   * 
   * if (cat.isInitialized()) {
   *   // `cat` is of type InitializedOneko
   * }
   * ```
   * 
   * @since 3.0.0
   */
  interface InitializedOneko extends EventTarget {
    /**
     * Controls if onAnimationFrame() loops after each completion of itself.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#loopanimating-boolean)
     * @since 2.2.0
     */
    loopAnimating: boolean;
    /**
     * Controls if the alert animation is skipped before running begins.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#skipalertanimation-boolean)
     * @since 2.2.0
     */
    skipAlertAnimation: boolean;
    /**
     * The Oneko's `element`'s position on the X axis, in pixels.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#x-number)
     * @since 1.0.0
     */
    x: number;
    /**
     * The Oneko's `element`'s position on the Y axis, in pixels.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#y-number)
     * @since 1.0.0
     */
    y: number;
    /**
     * How far the Oneko runs per update, in pixels.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#speed-number)
     * @since 1.0.0
     */
    speed: number;
    /**
     * How big the Oneko is, in pixels.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#size-number)
     * @since 3.0.0
     */
    size: number;
    /**
     * The maximum distance, in pixels, that the Oneko is allowed to be from the target point before becomes idle.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#allowedtargetdistance-number)
     * @since 3.0.0
     */
    allowedTargetDistance: number;
    /**
     * The path to an image file used to represent the Oneko.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#source-string)
     * @since 1.0.0
     */
    source: string;
    /**
     * How fast the Oneko updates its animations, in milliseconds.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#updatespeed-number)
     * @since 1.0.0
     */
    updateSpeed: number;
    /**
     * An HTMLDivElement used to represent the Oneko in the document.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#element-htmldivelement--null)
     * @since 1.0.0
     */
    element: HTMLDivElement | null;
    /**
     * The X position the Oneko is running towards, in pixels.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#targetx-number)
     * @since 1.0.0
     */
    targetX: number;
    /**
     * The Y position the Oneko is running towards, in pixels.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#targety-number)
     * @since 1.0.0
     */
    targetY: number;
    /**
     * How long the Oneko has been alive for. Measured by how many times the Oneko's `element` has been updated.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#framecount-number)
     * @since 1.0.0
     */
    frameCount: number;
    /**
     * How long the Oneko has been idle for. Measured by how many times the Oneko's `element` has been updated.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#idletime-number)
     * @since 1.0.0
     */
    idleTime: number;
    /**
     * The idle animation that's currently playing. `null` means the regular idle animation of being played.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#idleanimation-onekoidleanimation)
     * @since 1.0.0
     */
    idleAnimation: OnekoIdleAnimation | null;
    /**
     * The current frame of the playing idle animation.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#idleanimationframe-number)
     * @since 1.0.0
     */
    idleAnimationFrame: number;
    /**
     * The timestamp of the last time the Oneko's `element` was updated.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#lastframetimestamp-number)
     * @since 1.0.0
     */
    readonly lastFrameTimestamp: number | null;
    /**
     * A list of animations that the Oneko can use when idle.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#allowedidleanimations-onekoidleanimation)
     * @since 3.0.0
     */
    allowedIdleAnimations: OnekoIdleAnimation[];
    /**
     * The number of frames the Oneko will spend in the `tired` state before entering the main sleep animation.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#yawnduration-number)
     * @since 3.0.0
     */
    yawnDuration: number;
    /**
     * The number of frames the Oneko will spend in the main sleep animation before the idle animation is reset.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#sleepduration-number)
     * @since 3.0.0
     */
    sleepDuration: number;
    /**
     * The number of frames the Oneko will spend in any scratching animation before the idle animation is reset.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#scratchduration-number)
     * @since 3.0.0
     */
    scratchDuration: number;
    /**
     * The maximum number of frames the Oneko will spend in the `alert` state before moving towards the target point.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#maxalertduration-number)
     * @since 3.0.0
     */
    maxAlertDuration: number;
  }

  type Oneko = (InitializedOneko | UninitializedOneko) & OnekoCommons;

  /**
   * An Oneko that the constructor method attempted to initialize using the provided parameters. It may or may not be initialized.
   * 
   * If the type checker can verify that the Oneko was initialized (this can be done with methods like `force()` or `isInitialized()`), it will become an `InitializedOneko`. However, because it is currently unknown whether the Oneko is initialized or not, all attributes are possibly `undefined`.
   * 
   * ```ts
   * let cat = new Oneko();
   * 
   * window.addEventListener("mousemove", (event) => {
   *   cat.setTarget(event.clientX, event.clientY);
   * });
   * ```
   * 
   * @since 1.0.0
   */
  export const Oneko: {
    prototype: Oneko,
    new(
      /**
       * Allows for applying custom values to properties during initialization.
       */
      options?: Partial<Pick<InitializedOneko, { [K in keyof InitializedOneko]: InitializedOneko[K] extends Function ? never : K }[keyof InitializedOneko]>> & { skipElementInit?: boolean }
    ): Oneko,
    readonly sourceOptions: [
      "ace",
      "black",
      "bunny",
      "calico",
      "default",
      "eevee",
      "esmeralda",
      "fox",
      "ghost",
      "gray",
      "jess",
      "kina",
      "lucy",
      "maia",
      "maria",
      "mike",
      "silver",
      "silversky",
      "snuupy",
      "spirit",
      "tora",
      "valentine"
    ],
    /**
     * Constructs a complete URL path to a file of a given name on the online source database.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#createdatabasesourceurlsourcename-onekodatabasesource-string)
     * @since 3.0.0
     */
    createDatabaseSourceURL(
      /**
       * The name of the image to access from the source database.
       */
      sourceName: OnekoDatabaseSource | "random" | (string & {})
    ): `https://raw.githubusercontent.com/raynecloudy/oneko_db/refs/heads/master/${typeof sourceName}.png`,
    /**
     * Returns `true` if an Oneko can be initialized under current conditions. An example in which this would return `false` is if the `prefers-reduced-motion` media query is set to `reduce`.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#caninitialize-boolean)
     * @since 3.0.0
     */
    canInitialize(): boolean,
  };

  interface OnekoCommons {
    /**
     * A keyed list of arrays of points ([number, number]), defined as animations.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#spritesets)
     * 
     * ![image](https://raw.githubusercontent.com/raynecloudy/oneko_db/refs/heads/master/default.png)
     * @since 1.0.0
     */
    spriteSets: {
      /**
       * The Oneko is standing still.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/idle.png)
       * @since 1.0.0
       */
      idle: number[][],
      /**
       * Shown before the Oneko starts running to the target coordinate.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/alert.png)
       * @since 1.0.0
       */
      alert: number[][],
      /**
       * Alternative idle animation - the Oneko scratches its ears.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/scratchSelf.png)
       * @since 1.0.0
       */
      scratchSelf: number[][],
      /**
       * Alternative idle animation - the Oneko scratches the top of the viewport. Can only be seen when the Oneko's `element` is less than `size` pixels away from the top of the viewport.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/scratchWallN.png)
       * @since 1.0.0
       */
      scratchWallN: number[][],
      /**
       * Alternative idle animation - the Oneko scratches the bottom of the viewport. Can only be seen when the Oneko's `element` is less than `size` pixels away from the bottom of the viewport.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/scratchWallS.png)
       * @since 1.0.0
       */
      scratchWallS: number[][],
      /**
       * Alternative idle animation - the Oneko scratches the right side of the viewport. Can only be seen when the Oneko's `element` is less than `size` pixels away from the right side of the viewport.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/scratchWallE.png)
       * @since 1.0.0
       */
      scratchWallE: number[][],
      /**
       * Alternative idle animation - the Oneko scratches the left side of the viewport. Can only be seen when the Oneko's `element` is less than `size` pixels away from the left side of the viewport.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/scratchWallW.png)
       * @since 1.0.0
       */
      scratchWallW: number[][],
      /**
       * Alternative idle animation - the Oneko is getting ready to sleep.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/tired.png)
       * @since 1.0.0
       */
      tired: number[][],
      /**
       * Alternative idle animation - the Oneko is sleeping.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/sleeping.png)
       * @since 1.0.0
       */
      sleeping: number[][],
      /**
       * The Oneko is running up.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/N.png)
       * @since 1.0.0
       */
      N: number[][],
      /**
       * The Oneko is running up and to the right.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/NE.png)
       * @since 1.0.0
       */
      NE: number[][],
      /**
       * The Oneko is running right.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/E.png)
       * @since 1.0.0
       */
      E: number[][],
      /**
       * The Oneko is running down and to the right.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/SE.png)
       * @since 1.0.0
       */
      SE: number[][],
      /**
       * The Oneko is running down.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/S.png)
       * @since 1.0.0
       */
      S: number[][],
      /**
       * The Oneko is running down and to the left.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/SW.png)
       * @since 1.0.0
       */
      SW: number[][],
      /**
       * The Oneko is running left.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/W.png)
       * @since 1.0.0
       */
      W: number[][],
      /**
       * The Oneko is running up and to the left.
       * 
       * ![image](https://raw.githubusercontent.com/raynecloudy/lots-o-nekos/refs/heads/master/media/anim-locations/NW.png)
       * @since 1.0.0
       */
      NW: number[][]
    };
    /**
     * Sets the coordinates for the Oneko to run to.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#settargetx-number-y-number-oneko)
     * @since 1.0.0
     */
    setTarget(
      /**
       * X location, in pixels.
       */
      x: number,
      /**
       * Y location, in pixels.
       */
      y: number
    ): Oneko;
    /**
     * Sets the coordinates for the Oneko element to be positioned at.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#setpositionx-number-y-number-oneko)
     * @since 2.3.0
     */
    setPosition(
      /**
       * X location, in pixels.
       */
      x: number,
      /**
       * Y location, in pixels.
       */
      y: number
    ): Oneko;
    /**
     * Sets the Oneko's target coordinates and element position.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#movetox-number-y-number-oneko)
     * @since 2.3.0
     */
    moveTo(
      /**
       * X location, in pixels.
       */
      x: number,
      /**
       * Y location, in pixels.
       */
      y: number
    ): Oneko;
    /**
     * Sets the source image of the Oneko element to a URL accessing the source database of Oneko PNGs (https://github.com/raynecloudy/oneko_db/). Recently added images may not appear in the selector.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#setsourcedbsourcename-onekodatabasesource-oneko)
     * @since 2.1.0
     */
    setSourceDB(
      /**
       * The name of the image to access from the source database.
       */
      sourceName: OnekoDatabaseSource | "random" | (string & {})
    ): Oneko;
    /**
     * Runs every frame. Enables Oneko animations.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#onanimationframetimestamp-number-oneko)
     * @since 2.3.0
     * @throws {TypeError} If the Oneko is not initialized.
     */
    onAnimationFrame(
      /**
       * Duration since last update.
       */
      timestamp: number
    ): Oneko;
    /**
     * Sets the sprite image to a given frame of a given animation.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#setspritename-string-frame-number-oneko)
     * @since 2.3.0
     * @throws {TypeError} If the Oneko is not initialized.
     */
    setSprite(
      /**
       * Name of animation to access.
       */
      setName: OnekoSpriteSetOption,
      /**
       * Frame of animation to access.
       */
      frame: number
    ): Oneko;
    /**
     * Resets the idle animation.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#resetidleanimation-oneko)
     * @since 2.3.0
     */
    resetIdleAnimation(): Oneko;
    /**
     * Controls idle animation logic (scratching, sleeping, etc.)
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#idle-oneko)
     * @since 2.3.0
     * @throws {TypeError} If the Oneko is not initialized.
     */
    idle(): Oneko;
    /**
     * Controls all animation logic.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#frame-oneko)
     * @since 2.3.0
     * @throws {TypeError} If the Oneko is not initialized.
     */
    frame(): Oneko;
    /**
     * Updates the Oneko element's position and image. Fires the `draw` event after completion.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#draw-oneko)
     * @since 2.3.0
     * @throws {TypeError} If the Oneko is not initialized.
     */
    draw(): Oneko;
    /**
     * Only `true` if the Oneko has been properly initialized. For example, if the `prefers-reduced-motion` media query is set to `reduce`, the Oneko will not initialize.
     * 
     * ```ts
     * let cat = new Oneko();
     * 
     * if (cat.isInitialized()) {
     *   // `cat` was initialized
     * } else {
     *   // `cat` was not initialized
     * }
     * ```
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#isinitialized-boolean)
     * @since 3.0.0
     */
    isInitialized(): this is InitializedOneko;
    /**
     * Forces an initialized Oneko object. This is intended to be used in variable declaration, like shown below.
     * 
     * ```ts
     * let cat = new Oneko().force();
     * ```
     * 
     * Due to the possiblity of throwing an error, it is not recommended to use this in production without a try/catch block. Consider using `isInitialized()` instead.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#force-oneko)
     * @since 3.0.0
     * @throws {Error} If the Oneko cannot be initialized.
     */
    force(): Oneko;
    /**
     * Takes the attributes of the class and compiles them into a JSON-compatible object.
     * 
     * @since 3.0.0
     */
    toJSON(): Pick<InitializedOneko, { [K in keyof InitializedOneko]: InitializedOneko[K] extends Function ? never : K }[keyof InitializedOneko]>;

    addEventListener<K extends keyof OnekoEventMap>(type: K, listener: (this: Document, ev: OnekoEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;
    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;
    removeEventListener<K extends keyof OnekoEventMap>(type: K, listener: (this: Document, ev: OnekoEventMap[K]) => any, options?: boolean | EventListenerOptions): void;
    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;
  }

  type OnekoDatabaseSource = typeof Oneko.sourceOptions[number];

  interface OnekoEventMap {
    /**
     * Fired after the draw() method is finished.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#draw)
     * @since 1.1.0
     */
    draw: Event,
    /**
     * Fired after target coordinate becomes outside range, after alert animation plays.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#startrunning)
     * @since 2.0.0
     */
    startRunning: Event,
    /**
     * Fired after target coordinate becomes inside range.
     * 
     * [Documentation Reference](https://github.com/raynecloudy/lots-o-nekos/blob/master/DOCUMENTATION.md#stoprunning)
     * @since 2.0.0
     */
    stopRunning: Event
  }

  type OnekoIdleAnimation = "sleeping" | "scratchSelf" | "scratchWallW" | "scratchWallN" | "scratchWallE" | "scratchWallS";


  type OnekoSpriteSetOption = keyof typeof Oneko.prototype.spriteSets;

  /**
   * An Oneko that the constructor method failed to initialize using the provided parameters. This would happen if the `prefers-reduced-motion` media query is set to `reduce`.
   * 
   * ```ts
   * let cat = new Oneko();
   * 
   * if (!cat.isInitialized()) {
   *   // `cat` is of type UninitializedOneko
   * }
   * ```
   * 
   * @since 3.0.0
   */
  interface UninitializedOneko {
    loopAnimating: undefined;
    skipAlertAnimation: undefined;
    x: undefined;
    y: undefined;
    speed: undefined;
    size: undefined;
    allowedTargetDistance: undefined;
    source: undefined;
    updateSpeed: undefined;
    element: undefined;
    targetX: undefined;
    targetY: undefined;
    frameCount: undefined;
    idleTime: undefined;
    idleAnimation: undefined;
    idleAnimationFrame: undefined;
    readonly lastFrameTimestamp: undefined;
    allowedIdleAnimations: undefined;
    yawnDuration: undefined;
    sleepDuration: undefined;
    scratchDuration: undefined;
    maxAlertDuration: undefined;
  }
}
