// src/index.ts
import { BroadcastChannel } from "broadcast-channel";
function broadcastQueryClient({
  queryClient,
  broadcastChannel = "tanstack-query",
  options
}) {
  let transaction = false;
  const tx = (cb) => {
    transaction = true;
    cb();
    transaction = false;
  };
  const channel = new BroadcastChannel(broadcastChannel, {
    webWorkerSupport: false,
    ...options
  });
  const queryCache = queryClient.getQueryCache();
  const unsubscribe = queryClient.getQueryCache().subscribe((queryEvent) => {
    if (transaction) {
      return;
    }
    const {
      query: { queryHash, queryKey, state, observers }
    } = queryEvent;
    if (queryEvent.type === "updated" && queryEvent.action.type === "success") {
      channel.postMessage({
        type: "updated",
        queryHash,
        queryKey,
        state
      });
    }
    if (queryEvent.type === "removed" && observers.length > 0) {
      channel.postMessage({
        type: "removed",
        queryHash,
        queryKey
      });
    }
    if (queryEvent.type === "added") {
      channel.postMessage({
        type: "added",
        queryHash,
        queryKey
      });
    }
  });
  channel.onmessage = (action) => {
    if (!(action == null ? void 0 : action.type)) {
      return;
    }
    tx(() => {
      const { type, queryHash, queryKey, state } = action;
      const query = queryCache.get(queryHash);
      if (type === "updated") {
        if (query) {
          query.setState(state);
          return;
        }
        queryCache.build(
          queryClient,
          {
            queryKey,
            queryHash
          },
          state
        );
      } else if (type === "removed") {
        if (query) {
          queryCache.remove(query);
        }
      } else if (type === "added") {
        if (query) {
          query.setState(state);
          return;
        }
        queryCache.build(
          queryClient,
          {
            queryKey,
            queryHash
          },
          state
        );
      }
    });
  };
  return () => {
    unsubscribe();
    channel.close();
  };
}
export {
  broadcastQueryClient
};
//# sourceMappingURL=index.js.map