import { Store } from '@tanstack/store';
import { QueuePosition } from './queuer.cjs';
export interface AsyncQueuerState<TValue> {
    /**
     * Items currently being processed by the queuer
     */
    activeItems: Array<TValue>;
    /**
     * Number of task executions that have resulted in errors
     */
    errorCount: number;
    /**
     * Number of items that have been removed from the queue due to expiration
     */
    expirationCount: number;
    /**
     * Whether the queuer has no items to process (items array is empty)
     */
    isEmpty: boolean;
    /**
     * Whether the queuer has reached its maximum capacity
     */
    isFull: boolean;
    /**
     * Whether the queuer is not currently processing any items
     */
    isIdle: boolean;
    /**
     * Whether the queuer is active and will process items automatically
     */
    isRunning: boolean;
    /**
     * Array of items currently waiting to be processed
     */
    items: Array<TValue>;
    /**
     * Timestamps when items were added to the queue for expiration tracking
     */
    itemTimestamps: Array<number>;
    /**
     * The result from the most recent task execution
     */
    lastResult: any;
    /**
     * Whether the queuer has a pending timeout for processing the next item
     */
    pendingTick: boolean;
    /**
     * Number of items that have been rejected from being added to the queue
     */
    rejectionCount: number;
    /**
     * Number of task executions that have completed (either successfully or with errors)
     */
    settledCount: number;
    /**
     * Number of items currently in the queue
     */
    size: number;
    /**
     * Current processing status - 'idle' when not processing, 'running' when active, 'stopped' when paused
     */
    status: 'idle' | 'running' | 'stopped';
    /**
     * Number of task executions that have completed successfully
     */
    successCount: number;
}
export interface AsyncQueuerOptions<TValue> {
    /**
     * Default position to add items to the queuer
     * @default 'back'
     */
    addItemsTo?: QueuePosition;
    /**
     * Maximum number of concurrent tasks to process.
     * Can be a number or a function that returns a number.
     * @default 1
     */
    concurrency?: number | ((queuer: AsyncQueuer<TValue>) => number);
    /**
     * Maximum time in milliseconds that an item can stay in the queue
     * If not provided, items will never expire
     */
    expirationDuration?: number;
    /**
     * Function to determine if an item has expired
     * If provided, this overrides the expirationDuration behavior
     */
    getIsExpired?: (item: TValue, addedAt: number) => boolean;
    /**
     * Default position to get items from during processing
     * @default 'front'
     */
    getItemsFrom?: QueuePosition;
    /**
     * Function to determine priority of items in the queuer
     * Higher priority items will be processed first
     * If not provided, will use static priority values attached to tasks
     */
    getPriority?: (item: TValue) => number;
    /**
     * Initial items to populate the queuer with
     */
    initialItems?: Array<TValue>;
    /**
     * Initial state for the async queuer
     */
    initialState?: Partial<AsyncQueuerState<TValue>>;
    /**
     * Maximum number of items allowed in the queuer
     */
    maxSize?: number;
    /**
     * Optional error handler for when a task throws.
     * If provided, the handler will be called with the error and queuer instance.
     * This can be used alongside throwOnError - the handler will be called before any error is thrown.
     */
    onError?: (error: unknown, item: TValue, queuer: AsyncQueuer<TValue>) => void;
    /**
     * Callback fired whenever an item expires in the queuer
     */
    onExpire?: (item: TValue, queuer: AsyncQueuer<TValue>) => void;
    /**
     * Callback fired whenever an item is added or removed from the queuer
     */
    onItemsChange?: (queuer: AsyncQueuer<TValue>) => void;
    /**
     * Callback fired whenever an item is rejected from being added to the queuer
     */
    onReject?: (item: TValue, queuer: AsyncQueuer<TValue>) => void;
    /**
     * Optional callback to call when a task is settled
     */
    onSettled?: (item: TValue, queuer: AsyncQueuer<TValue>) => void;
    /**
     * Optional callback to call when a task succeeds
     */
    onSuccess?: (result: any, item: TValue, queuer: AsyncQueuer<TValue>) => void;
    /**
     * Whether the queuer should start processing tasks immediately or not.
     */
    started?: boolean;
    /**
     * Whether to throw errors when they occur.
     * Defaults to true if no onError handler is provided, false if an onError handler is provided.
     * Can be explicitly set to override these defaults.
     */
    throwOnError?: boolean;
    /**
     * Time in milliseconds to wait between processing items.
     * Can be a number or a function that returns a number.
     * @default 0
     */
    wait?: number | ((queuer: AsyncQueuer<TValue>) => number);
}
/**
 * A flexible asynchronous queue for processing tasks with configurable concurrency, priority, and expiration.
 *
 * Features:
 * - Priority queue support via the getPriority option
 * - Configurable concurrency limit
 * - Callbacks for task success, error, completion, and queue state changes
 * - FIFO (First In First Out) or LIFO (Last In First Out) queue behavior
 * - Pause and resume processing
 * - Task cancellation
 * - Item expiration to remove stale items from the queue
 *
 * Tasks are processed concurrently up to the configured concurrency limit. When a task completes,
 * the next pending task is processed if the concurrency limit allows.
 *
 * Error Handling:
 * - If an `onError` handler is provided, it will be called with the error and queuer instance
 * - If `throwOnError` is true (default when no onError handler is provided), the error will be thrown
 * - If `throwOnError` is false (default when onError handler is provided), the error will be swallowed
 * - Both onError and throwOnError can be used together; the handler will be called before any error is thrown
 * - The error state can be checked using the AsyncQueuer instance
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the async queuer
 * - Use `onSuccess` callback to react to successful task execution and implement custom logic
 * - Use `onError` callback to react to task execution errors and implement custom error handling
 * - Use `onSettled` callback to react to task execution completion (success or error) and implement custom logic
 * - Use `onItemsChange` callback to react to items being added or removed from the queue
 * - Use `onExpire` callback to react to items expiring and implement custom logic
 * - Use `onReject` callback to react to items being rejected when the queue is full
 * - The state includes error count, expiration count, rejection count, running status, and success/settle counts
 * - State can be accessed via `asyncQueuer.store.state` when using the class directly
 * - When using framework adapters (React/Solid), state is accessed from `asyncQueuer.state`
 *
 * Example usage:
 * ```ts
 * const asyncQueuer = new AsyncQueuer<string>(async (item) => {
 *   // process item
 *   return item.toUpperCase();
 * }, {
 *   concurrency: 2,
 *   onSuccess: (result) => {
 *     console.log(result);
 *   }
 * });
 *
 * asyncQueuer.addItem('hello');
 * asyncQueuer.start();
 * ```
 */
export declare class AsyncQueuer<TValue> {
    #private;
    fn: (item: TValue) => Promise<any>;
    readonly store: Store<Readonly<AsyncQueuerState<TValue>>>;
    options: AsyncQueuerOptions<TValue>;
    constructor(fn: (item: TValue) => Promise<any>, initialOptions?: AsyncQueuerOptions<TValue>);
    /**
     * Updates the queuer options. New options are merged with existing options.
     */
    setOptions: (newOptions: Partial<AsyncQueuerOptions<TValue>>) => void;
    /**
     * Adds an item to the queue. If the queue is full, the item is rejected and onReject is called.
     * Items can be inserted based on priority or at the front/back depending on configuration.
     *
     * @example
     * ```ts
     * queuer.addItem({ value: 'task', priority: 10 });
     * queuer.addItem('task2', 'front');
     * ```
     */
    addItem: (item: TValue, position?: QueuePosition, runOnItemsChange?: boolean) => boolean;
    /**
     * Removes and returns the next item from the queue without executing the task function.
     * Use for manual queue management. Normally, use execute() to process items.
     *
     * @example
     * ```ts
     * // FIFO
     * queuer.getNextItem();
     * // LIFO
     * queuer.getNextItem('back');
     * ```
     */
    getNextItem: (position?: QueuePosition) => TValue | undefined;
    /**
     * Removes and returns the next item from the queue and executes the task function with it.
     *
     * @example
     * ```ts
     * queuer.execute();
     * // LIFO
     * queuer.execute('back');
     * ```
     */
    execute: (position?: QueuePosition) => Promise<any>;
    /**
     * Processes a specified number of items to execute immediately with no wait time
     * If no numberOfItems is provided, all items will be processed
     */
    flush: (numberOfItems?: number, position?: QueuePosition) => Promise<void>;
    /**
     * Processes all items in the queue as a batch using the provided function as an argument
     * The queue is cleared after processing
     */
    flushAsBatch: (batchFunction: (items: Array<TValue>) => Promise<any>) => Promise<void>;
    /**
     * Returns the next item in the queue without removing it.
     *
     * @example
     * ```ts
     * queuer.peekNextItem(); // front
     * queuer.peekNextItem('back'); // back
     * ```
     */
    peekNextItem: (position?: QueuePosition) => TValue | undefined;
    /**
     * Returns a copy of all items in the queue, including active and pending items.
     */
    peekAllItems: () => Array<TValue>;
    /**
     * Returns the items currently being processed (active tasks).
     */
    peekActiveItems: () => Array<TValue>;
    /**
     * Returns the items waiting to be processed (pending tasks).
     */
    peekPendingItems: () => Array<TValue>;
    /**
     * Starts processing items in the queue. If already running, does nothing.
     */
    start: () => void;
    /**
     * Stops processing items in the queue. Does not clear the queue.
     */
    stop: () => void;
    /**
     * Removes all pending items from the queue. Does not affect active tasks.
     */
    clear: () => void;
    /**
     * Resets the queuer state to its default values
     */
    reset: () => void;
}
/**
 * Creates a new AsyncQueuer instance and returns a bound addItem function for adding tasks.
 * The queuer is started automatically and ready to process items.
 *
 * Error Handling:
 * - If an `onError` handler is provided, it will be called with the error and queuer instance
 * - If `throwOnError` is true (default when no onError handler is provided), the error will be thrown
 * - If `throwOnError` is false (default when onError handler is provided), the error will be swallowed
 * - Both onError and throwOnError can be used together; the handler will be called before any error is thrown
 * - The error state can be checked using the underlying AsyncQueuer instance
 *
 * State Management:
 * - Uses TanStack Store for reactive state management
 * - Use `initialState` to provide initial state values when creating the async queuer
 * - Use `onSuccess` callback to react to successful task execution and implement custom logic
 * - Use `onError` callback to react to task execution errors and implement custom error handling
 * - Use `onSettled` callback to react to task execution completion (success or error) and implement custom logic
 * - Use `onItemsChange` callback to react to items being added or removed from the queue
 * - Use `onExpire` callback to react to items expiring and implement custom logic
 * - Use `onReject` callback to react to items being rejected when the queue is full
 * - The state includes error count, expiration count, rejection count, running status, and success/settle counts
 * - State can be accessed via the underlying AsyncQueuer instance's `store.state` property
 * - When using framework adapters (React/Solid), state is accessed from the hook's state property
 *
 * Example usage:
 * ```ts
 * const enqueue = asyncQueue<string>(async (item) => {
 *   return item.toUpperCase();
 * }, {...options});
 *
 * enqueue('hello');
 * ```
 */
export declare function asyncQueue<TValue>(fn: (value: TValue) => Promise<any>, initialOptions: AsyncQueuerOptions<TValue>): (item: TValue, position?: QueuePosition, runOnItemsChange?: boolean) => boolean;
