import { ContainerBoxes } from './ContainerBoxes.js';
import { DATA } from './fields/DATA.js';
import { INT } from './fields/INT.js';
import { STRING } from './fields/STRING.js';
import { TEMPLATE } from './fields/TEMPLATE.js';
import { UINT } from './fields/UINT.js';
import { UTF8 } from './fields/UTF8.js';
import { readData } from './readers/readData.js';
import { readInt } from './readers/readInt.js';
import { readString } from './readers/readString.js';
import { readTemplate } from './readers/readTemplate.js';
import { readTerminatedString } from './readers/readTerminatedString.js';
import { readUint } from './readers/readUint.js';
import { readUTF8String } from './readers/readUTF8String.js';
import { readUTF8TerminatedString } from './readers/readUTF8TerminatedString.js';
/**
 * ISO BMFF data view. Similar to DataView, but with additional methods for reading ISO BMFF data.
 * It implements the iterator protocol, so it can be used in a for...of loop.
 *
 * @group ISOBMFF
 *
 * @beta
 */
export class IsoView {
    constructor(raw, config) {
        this.truncated = false;
        this.slice = (size) => {
            const dataView = new DataView(this.dataView.buffer, this.offset, size);
            this.offset += size;
            return new IsoView(dataView, this.config);
        };
        this.read = (type, size = 0) => {
            // TODO: Change all sizes from bits to bytes
            const { dataView, offset } = this;
            let result;
            let cursor = size;
            switch (type) {
                case UINT:
                    result = readUint(dataView, offset, size);
                    break;
                case INT:
                    result = readInt(dataView, offset, size);
                    break;
                case TEMPLATE:
                    result = readTemplate(dataView, offset, size);
                    break;
                case STRING:
                    if (size === -1) {
                        result = readTerminatedString(dataView, offset);
                        cursor = result.length + 1;
                    }
                    else {
                        result = readString(dataView, offset, size);
                    }
                    break;
                case DATA:
                    result = readData(dataView, offset, size);
                    cursor = result.length;
                    break;
                case UTF8:
                    if (size === -1) {
                        result = readUTF8TerminatedString(dataView, offset);
                        cursor = result.length + 1;
                    }
                    else {
                        result = readUTF8String(dataView, offset);
                    }
                    break;
                default:
                    result = -1;
            }
            this.offset += cursor;
            return result;
        };
        this.readUint = (size) => {
            return this.read(UINT, size);
        };
        this.readInt = (size) => {
            return this.read(INT, size);
        };
        this.readString = (size) => {
            return this.read(STRING, size);
        };
        this.readTemplate = (size) => {
            return this.read(TEMPLATE, size);
        };
        this.readData = (size) => {
            return this.read(DATA, size);
        };
        this.readUtf8 = (size) => {
            return this.read(UTF8, size);
        };
        this.readFullBox = () => {
            return {
                version: this.readUint(1),
                flags: this.readUint(3),
            };
        };
        this.readArray = (type, size, length) => {
            const value = [];
            for (let i = 0; i < length; i++) {
                value.push(this.read(type, size));
            }
            return value;
        };
        this.readBox = () => {
            const { dataView, offset } = this;
            // read box size and type without advancing the cursor in case the box is truncated
            let cursor = 0;
            const box = {
                size: readUint(dataView, offset, 4),
                type: readString(dataView, offset + 4, 4),
            };
            cursor += 8;
            if (box.size === 1) {
                box.largesize = readUint(dataView, offset + cursor, 8);
                cursor += 8;
            }
            const actualSize = box.largesize || box.size;
            if (this.cursor + actualSize > dataView.byteLength) {
                this.truncated = true;
                throw new Error('Truncated box');
            }
            this.offset += cursor;
            if (box.type === 'uuid') {
                box.usertype = this.readArray('uint', 1, 16);
            }
            const viewSize = box.size === 0 ? this.bytesRemaining : actualSize - cursor;
            box.data = this.slice(viewSize);
            return box;
        };
        this.readBoxes = (length) => {
            const result = [];
            for (const box of this) {
                result.push(box);
                if (length > 0 && result.length >= length) {
                    break;
                }
            }
            return result;
        };
        this.readEntries = (length, map) => {
            const result = [];
            for (let i = 0; i < length; i++) {
                result.push(map());
            }
            return result;
        };
        this.dataView = (raw instanceof ArrayBuffer) ? new DataView(raw) : (raw instanceof Uint8Array) ? new DataView(raw.buffer, raw.byteOffset, raw.byteLength) : raw;
        this.offset = this.dataView.byteOffset;
        this.config = config || { recursive: false, parsers: {} };
    }
    get cursor() {
        return this.offset - this.dataView.byteOffset;
    }
    get done() {
        return this.cursor >= this.dataView.byteLength || this.truncated;
    }
    get bytesRemaining() {
        return this.dataView.byteLength - this.cursor;
    }
    *[Symbol.iterator]() {
        const { parsers = {}, recursive = false } = this.config;
        while (!this.done) {
            try {
                const { type, data, ...rest } = this.readBox();
                const box = { type, ...rest };
                const parser = parsers[type] || parsers[type.trim()]; // url and urn boxes have a trailing space in their type field
                if (parser) {
                    Object.assign(box, parser(data, this.config));
                }
                if (ContainerBoxes.includes(type)) {
                    const boxes = [];
                    for (const child of data) {
                        if (recursive) {
                            yield child;
                        }
                        boxes.push(child);
                    }
                    box.boxes = boxes;
                }
                yield box;
            }
            catch (error) {
                break;
            }
        }
    }
}
//# sourceMappingURL=IsoView.js.map