// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.units

import android.content.Context
import xyz.apiote.bimba.czwek.R
import java.lang.Long.MAX_VALUE
import java.lang.Long.MIN_VALUE
import java.text.NumberFormat
import kotlin.math.abs

object Metric : UnitSystem(10) {

	override fun timeUnit(count: Long): TimeUnit = Second(count.toDouble())
	override fun timeUnit(other: TimeUnit): TimeUnit = Second(other)

	override fun speedUnit(count: Double): SpeedUnit = Kmph(count)
	override fun speedUnit(other: SpeedUnit): SpeedUnit = Kmph(other)

	override fun distanceUnit(count: Double): DistanceUnit = Km(count)
	override fun distanceUnit(other: DistanceUnit): DistanceUnit = Km(other)

	override fun toString(context: Context, s: SpeedUnit): String = s.toString(context, base)
	override fun toString(context: Context, t: TimeUnit, relative: Boolean) =
		FriendlyDuration(t).toString(context, base, relative)

	override fun toString(context: Context, d: DistanceUnit): String = Km(d).toString(context, base)
}

class Km(val km: Double) : DistanceUnit {
	constructor(other: DistanceUnit) : this(other.meters() / 1000)

	override fun meters(): Double = km * 1000
	override fun toString(context: Context, base: Int): String =
		if (km < 1) {
			context.getString(
				R.string.distance_in_m,
				NumberFormat.getIntegerInstance().format((km * 1000).toInt())
			)
		} else {
			context.getString(
				R.string.distance_in_km,
				NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(km)
			)
		}

	override fun contentDescription(context: Context, base: Int): String =
		if (km < 1) {
			context.resources.getQuantityString(R.plurals.distance_in_m_cd, (km * 1000).toInt())
		} else {
			val kilometres = km.toInt()
			val metres = ((km - kilometres) * 1000).toInt()
			val kmString = context.resources.getQuantityString(R.plurals.distance_in_km_cd, kilometres)
			val mString = context.resources.getQuantityString(R.plurals.distance_in_m_cd, metres)
			if (metres > 0) {
				context.getString(R.string.distance_in_two_units_cd, kmString, mString)
			} else {
				kmString
			}
		}
}

class Metre(val m: Double) : DistanceUnit {
	constructor(other: DistanceUnit) : this(other.meters())

	override fun meters(): Double = m

	override fun toString(context: Context, base: Int): String =
		context.getString(R.string.distance_in_m, NumberFormat.getIntegerInstance().format(m.toInt()))

	override fun contentDescription(context: Context, base: Int): String =
		context.resources.getQuantityString(R.plurals.distance_in_m_cd, m.toInt(), m.toInt())
}

class Second(val seconds: Double) : TimeUnit {
	constructor(s: Int) : this(s.toDouble())
	constructor(other: TimeUnit) : this((other.milliseconds().toDouble() / 1000))

	override fun milliseconds(): Long = (seconds * 1000).toLong()
	override fun toString(context: Context, base: Int, relative: Boolean): String =
		context.getString(
			R.string.time_in_s,
			NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(seconds)
		)

	override fun contentDescription(context: Context, base: Int, relative: Boolean): String =
		context.resources.getQuantityString(R.plurals.time_in_s_cd, seconds.toInt(), seconds.toInt())
}

class FriendlyDuration(val minutes: Double) : TimeUnit {
	constructor(other: TimeUnit) : this((other.milliseconds().toDouble() / 1000 / 60))

	override fun milliseconds(): Long {
		return minutes.toLong() * 60 * 1000
	}

	override fun toString(context: Context, base: Int, relative: Boolean): String {
		val h = abs(milliseconds() / 1000 / 3600)
		val m_s = abs((milliseconds() / 1000) % 3600)
		val m = m_s / 60
		return if (h == 0L) {
			val res = if (relative) {
				when (milliseconds()) {
					in MIN_VALUE..<0 -> R.string.time_in_min_past
					in 0..<1 -> R.string.time_in_min_future_less_than_one
					in 1..MAX_VALUE -> R.string.time_in_min_future
					else -> throw IllegalStateException("${milliseconds()} is not real")
				}
			} else {
				R.string.time_in_min
			}
			context.getString(
				res,
				NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(m)
			)
		} else {
			val res = if (relative) {
				if (milliseconds() < 0) {
					R.string.time_in_h_min_past
				} else {
					R.string.time_in_h_min_future
				}
			} else {
				R.string.time_in_h_min
			}
			context.getString(
				res,
				NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(h),
				NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(m)
			)
		}
	}

	override fun contentDescription(
		context: Context,
		base: Int,
		relative: Boolean
	): String {
		val h = abs(milliseconds() / 1000 / 3600)
		val m_s = abs((milliseconds() / 1000) % 3600)
		val m = m_s / 60
		return if (h == 0L) {
			val res = if (relative) {
				when (milliseconds()) {
					in MIN_VALUE..<0 -> R.string.time_in_min_past_cd
					in 0..<1 -> R.string.time_in_min_future_less_than_one_cd
					in 1..MAX_VALUE -> R.string.time_in_min_future_cd
					else -> throw IllegalStateException("${milliseconds()} is not real")
				}
			} else {
				R.string.time_in_min_cd
			}
			context.getString(
				res,
				NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(m),
			)
		} else {
			val res = if (relative) {
				if (milliseconds() < 0) {
					R.string.time_in_h_min_past_cd
				} else {
					R.string.time_in_h_min_future_cd
				}
			} else {
				R.string.time_in_h_min_cd
			}
			val hoursCd = context.resources.getQuantityString(R.plurals.time_in_hours, h.toInt(), h)
			val minutesCd = context.resources.getQuantityString(R.plurals.time_in_minutes, m.toInt(), m)
			context.getString(
				res,
				hoursCd, minutesCd
			)
		}
	}
}

class Mps(val mps: Double) : SpeedUnit {
	constructor(s: Int) : this(s.toDouble())
	constructor(other: SpeedUnit) : this(other.mps())

	override fun mps(): Double = mps
	override fun toString(context: Context, base: Int): String =
		context.getString(
			R.string.speed_in_m_per_s,
			NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(mps)
		)

	override fun contentDescription(context: Context, base: Int): String =
		context.resources.getQuantityString(R.plurals.speed_in_m_per_s_cd, mps.toInt(), mps)
}

class Kmph(val kmph: Double) : SpeedUnit {
	constructor(s: Int) : this(s.toDouble())
	constructor(other: SpeedUnit) : this(other.mps() * 3.6)

	override fun mps(): Double = kmph / 3.6
	override fun toString(context: Context, base: Int): String =
		context.getString(
			R.string.speed_in_km_per_h,
			NumberFormat.getInstance().apply { maximumFractionDigits = 2 }.format(kmph)
		)

	override fun contentDescription(context: Context, base: Int): String =
		context.resources.getQuantityString(R.plurals.speed_in_km_per_h_cd, kmph.toInt(), kmph.toInt())
}