// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.settings.feeds

import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.View
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.isVisible
import androidx.core.view.updatePadding
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.transition.TransitionManager
import androidx.work.OneTimeWorkRequest
import androidx.work.WorkManager
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.google.android.material.shape.MaterialShapeDrawable
import com.google.android.material.snackbar.Snackbar
import com.google.android.material.transition.MaterialFade
import kotlinx.coroutines.launch
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.api.Error
import xyz.apiote.bimba.czwek.data.exceptions.BimbaException
import xyz.apiote.bimba.czwek.data.settings.SettingsRepository
import xyz.apiote.bimba.czwek.databinding.ActivityFeedChooserBinding
import xyz.apiote.bimba.czwek.repo.FeedInfo
import xyz.apiote.bimba.czwek.repo.User
import xyz.apiote.bimba.czwek.settings.DownloadCitiesWorker

// TODO on internet connection -> getServer
// TODO swipe to refresh?

// TODO if got Traffic after it was null -> copy FeedsSettings from EMPTY Traffic

class FeedChooserActivity : AppCompatActivity() {
	private lateinit var viewModel: FeedsViewModel
	private var _binding: ActivityFeedChooserBinding? = null
	private val binding get() = _binding!!

	private lateinit var adapter: BimbaFeedInfoAdapter

	override fun onCreate(savedInstanceState: Bundle?) {
		enableEdgeToEdge()
		super.onCreate(savedInstanceState)
		_binding = ActivityFeedChooserBinding.inflate(layoutInflater)
		setContentView(binding.root)

		binding.appBarLayout.statusBarForeground =
			MaterialShapeDrawable.createWithElevationOverlay(this)

		ViewCompat.setOnApplyWindowInsetsListener(binding.root) { v, windowInsets ->
			windowInsets.displayCutout?.safeInsetRight?.let { binding.resultsRecycler.updatePadding(right = it) }
			windowInsets.displayCutout?.safeInsetLeft?.let { binding.resultsRecycler.updatePadding(left = it) }
			val insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars())
			v.updatePadding(
				right = insets.right,
				left = insets.left,
				bottom = insets.bottom
			)
			windowInsets
		}

		binding.loading.loading.visibility = View.VISIBLE
		binding.resultsRecycler.visibility = View.GONE

		viewModel = ViewModelProvider(this)[FeedsViewModel::class.java]
		viewModel.getServer(this)

		setUpRecycler()

		lifecycleScope.launch {
			repeatOnLifecycle(Lifecycle.State.STARTED) {
				viewModel.settings.collect { settings ->
					if (settings.isEmpty()) {
						return@collect
					}

					updateItems(null, settings, viewModel.consumeNotifyRecycler())
				}
			}
		}

		lifecycleScope.launch {
			repeatOnLifecycle(Lifecycle.State.STARTED) {
				viewModel.feeds.collect { feeds ->
					if (feeds.isEmpty() && !viewModel.finishedGettingFeeds) {
						return@collect
					}

					updateItems(feeds.map { it.value }, null)
					if (viewModel.finishedGettingFeeds) {
						val errors = viewModel.consumeErrors()
						val error = when (errors.size) {
							0 -> null
							1 -> errors[0].cause
							else -> Error(0, R.string.error_multiple, R.drawable.error_other)
						}
						if (adapter.isNullOrEmpty()) {
							showError(error ?: Error(0, R.string.error_404, R.drawable.error_search), errors)
						} else if (error != null) {
							showErrorSnackbar(error, errors)
						}
					}
				}
			}
		}

		binding.buttonSave.setOnClickListener {
			moveOn()
		}

		binding.filterInput.addTextChangedListener(object : TextWatcher {
			override fun afterTextChanged(s: Editable?) {
				adapter.update(viewModel.feeds.value.values.toList().filter {
					binding.filterInput.text.toString().lowercase() in it.name.lowercase()
				}, null, true)
			}

			override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {}
			override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {}
		})

		val user = User.load(this)
		binding.topAppBar.menu.findItem(R.id.server).isVisible =
			viewModel.server.traffic.servers.filter { user.seat.toString() in it.seatsRequired }.size > 1

		binding.topAppBar.setOnMenuItemClickListener { menuItem ->
			when (menuItem.itemId) {
				R.id.server -> {
					val availableFlavours =
						viewModel.server.traffic.servers.filter { user.seat.toString() in it.seatsRequired }
							.map { it.description }
							.toTypedArray()
					val selected = availableFlavours.toList()
						.indexOf(viewModel.server.traffic.servers[viewModel.server.traffic.selectedServer].description)

					MaterialAlertDialogBuilder(this)
						.setTitle(resources.getString(R.string.title_select_server_flavour))
						.setSingleChoiceItems(availableFlavours, selected) { _, which ->
							if (which != -1) {
								viewModel.server.traffic.selectedServer = which
								viewModel.server.save(this)
								refreshFeeds()
							}
						}
						.show()
					true
				}

				R.id.filter -> {
					val feeds = if (binding.filterInput.isVisible) {
						binding.filterInput.visibility = View.GONE
						viewModel.feeds.value.values.toList()
					} else {
						binding.filterInput.visibility = View.VISIBLE
						viewModel.feeds.value.values.toList().filter {
							binding.filterInput.text.toString() in it.name
						}
					}
					adapter.update(feeds, null, true)
					true
				}

				else -> false
			}
		}
	}

	private fun showBottomSheet(feedID: String) {
		FeedBottomSheet(
			feedID,
			viewModel.feeds.value,
			viewModel.settings.value
		) { settings ->
			if (settings != null) {
				val fade = MaterialFade().apply {
					addTarget(R.id.feed_switch_container)
					duration = 500L
				}
				TransitionManager.beginDelayedTransition(binding.root, fade)
				viewModel.setSettings(feedID, settings)
				updateItems(null, viewModel.settings.value, false)
				adapter.notifyItemChanged(adapter.getItemPosition(feedID))
			}
		}.show(supportFragmentManager, FeedBottomSheet.TAG)
	}

	private fun setUpRecycler() {
		binding.resultsRecycler.layoutManager = LinearLayoutManager(this)
		adapter =
			BimbaFeedInfoAdapter(
				layoutInflater,
				emptyList(),
				FeedsSettings.EMPTY,
				{
					showBottomSheet(it)
				},
				{ feedID, isEnabled -> viewModel.setEnabled(feedID, isEnabled) }
			)
		binding.resultsRecycler.adapter = adapter
	}

	private fun moveOn() {
		viewModel.server.save(this)
		SettingsRepository().saveFeedsSettings(viewModel.settings.value, this)

		if (viewModel.settings.value.geocoding?.autoUpdate == true && DownloadCitiesWorker.shouldUpdate(
				this
			)
		) {
			WorkManager.getInstance(this)
				.enqueue(OneTimeWorkRequest.from(DownloadCitiesWorker::class.java))
		}
		finish()
	}

	private fun showError(error: Error, exceptions: List<BimbaException>) {
		binding.loading.loading.visibility = View.GONE
		binding.resultsRecycler.visibility = View.GONE
		binding.errorImage.apply {
			visibility = View.VISIBLE
			setImageDrawable(
				AppCompatResources.getDrawable(
					this@FeedChooserActivity,
					error.imageResource
				)
			)
		}
		binding.errorText.apply {
			visibility = View.VISIBLE
			setText(error.stringResource)
		}

		binding.moreButton.apply {
			visibility = View.VISIBLE
			setOnClickListener {
				BimbaException.showDialog(this@FeedChooserActivity, exceptions)
			}
		}
	}

	private fun showErrorSnackbar(error: Error, exceptions: List<BimbaException>) {
		val snackbar = Snackbar.make(binding.root, error.stringResource, Snackbar.LENGTH_LONG)
		snackbar.setAction(R.string.more_info) {
			BimbaException.showDialog(this, exceptions)
		}
		snackbar.show()
	}

	private fun refreshFeeds() {
		binding.loading.loading.visibility = View.VISIBLE
		binding.resultsRecycler.visibility = View.GONE
		viewModel.refresh(this)
	}

	private fun updateItems(
		feeds: List<FeedInfo>?,
		feedsSettings: FeedsSettings?,
		notify: Boolean = true
	) {
		binding.feedsOverlay.visibility = View.GONE
		binding.resultsRecycler.visibility = View.VISIBLE
		binding.buttonSave.visibility = View.VISIBLE
		adapter.update(feeds, feedsSettings, notify)
		if (feeds?.isEmpty() == true) {
			showError(Error(0, R.drawable.error_search, R.string.error_404), emptyList())
		}
	}
}