// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.repo

import android.content.Context
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Matrix
import android.graphics.Paint
import android.graphics.Path
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.graphics.createBitmap
import androidx.core.graphics.drawable.toBitmap
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.dpToPixel
import xyz.apiote.bimba.czwek.dpToPixelI
import kotlin.math.abs
import kotlin.math.cbrt
import kotlin.math.pow

interface LineAbstract {
	fun textColour(c: Colour): Int {
		val black = relativeLuminance(Colour(0u, 0u, 0u)) + .05
		val white = relativeLuminance(Colour(255u, 255u, 255u)) + .05
		val colour = relativeLuminance(c) + .05
		return if ((white / colour) > (colour / black)) {
			Color.WHITE
		} else {
			Color.BLACK
		}
	}

	private fun relativeLuminance(colour: Colour): Double {
		val r = fromSRGB(colour.R.toDouble() / 0xff)
		val g = fromSRGB(colour.G.toDouble() / 0xff)
		val b = fromSRGB(colour.B.toDouble() / 0xff)
		return 0.2126 * r + 0.7152 * g + 0.0722 * b
	}

	private fun fromSRGB(part: Double): Double {
		return if (part <= 0.03928) {
			part / 12.92
		} else {
			((part + 0.055) / 1.055).pow(2.4)
		}
	}

	fun icon(context: Context, type: LineType, colour: Colour, scale: Float): Bitmap {
		val drawingBitmap = createBitmap(dpToPixelI(24f / scale), dpToPixelI(24f / scale))
		val canvas = Canvas(drawingBitmap)

		canvas.drawPath(
			getSquirclePath(
				dpToPixel(.8f / scale), dpToPixel(.8f / scale), dpToPixelI(11.2f / scale)
			), Paint().apply { color = textColour(colour) })
		canvas.drawPath(
			getSquirclePath(
				dpToPixel(1.6f / scale), dpToPixel(1.6f / scale), dpToPixelI(10.4f / scale)
			), Paint().apply { color = colour.toInt() })

		val iconID = when (type) {
			LineType.WALK -> R.drawable.walk_black
			LineType.MISC -> R.drawable.vehicle_black
			LineType.TRAM -> R.drawable.tram_black
			LineType.METRO_UNDERGROUND -> R.drawable.underground_black
			LineType.TRAIN -> R.drawable.train_black
			LineType.BUS -> R.drawable.bus_black
			LineType.FERRY -> R.drawable.ferry_black
			LineType.CABLE_TRAM -> R.drawable.cable_tram_black
			LineType.CABLE_RAIL -> R.drawable.aerial_rail_black
			LineType.FUNICULAR -> R.drawable.funicular_black
			LineType.TROLLEYBUS -> R.drawable.trolleybus_black
			LineType.MONORAIL -> R.drawable.monorail_black
			LineType.TRAIN_RAPID -> R.drawable.train_rapid_black
			LineType.TRAIN_DISTANCE -> R.drawable.train_distance_black
			LineType.TRAIN_INTERREGIO -> R.drawable.train_black  // TODO
			LineType.TRAIN_CAR_TRANSPORT -> R.drawable.train_black  // TODO
			LineType.TRAIN_SLEEPER -> R.drawable.train_night_black
			LineType.TRAIN_REGIO -> R.drawable.train_black  // TODO
			LineType.TRAIN_TOURIST -> R.drawable.train_black  // TODO
			LineType.TRAIN_SHUTTLE -> R.drawable.train_black  // TODO
			LineType.TRAIN_SUBURBAN -> R.drawable.train_light_black
			LineType.TRAIN_REPLACEMENT -> R.drawable.train_black  // TODO
			LineType.TRAIN_SPECIAL -> R.drawable.train_black  // TODO
			LineType.TRAIN_LORRY_TRANSPORT -> R.drawable.train_black  // TODO
			LineType.TRAIN_ALL -> R.drawable.train_black  // TODO
			LineType.TRAIN_CROSS_COUNTRY -> R.drawable.train_black  // TODO
			LineType.TRAIN_VEHICLE_TRANSPORT -> R.drawable.train_black  // TODO
			LineType.TRAIN_RACK -> R.drawable.train_black  // TODO
			LineType.TRAIN_ADDITIONAL -> R.drawable.train_black  // TODO
			LineType.AIR -> R.drawable.plane_black
			LineType.COACH -> R.drawable.coach_black
			LineType.COACH_CROSS_COUNTRY -> R.drawable.coach_black // TODO
			LineType.COACH_INTERREGIO -> R.drawable.coach_black // TODO
			LineType.COACH_SHUTTLE -> R.drawable.coach_black // TODO
			LineType.COACH_REGIO -> R.drawable.coach_black // TODO
			LineType.COACH_SPECIAL -> R.drawable.coach_black // TODO
			LineType.COACH_SIGHTSEEING -> R.drawable.coach_black // TODO
			LineType.COACH_TOURIST -> R.drawable.coach_black // TODO
			LineType.COACH_COMMUTER -> R.drawable.coach_black // TODO
			LineType.COACH_ALL -> R.drawable.coach_black // TODO
			LineType.URBAN -> R.drawable.urban_black
			LineType.URBAN_METRO -> R.drawable.metro_black
			LineType.URBAN_UNDERGROUND -> R.drawable.underground_black
			LineType.URBAN_RAIL -> R.drawable.urban_black
			LineType.URBAN_ALL -> R.drawable.vehicle_black //TODO
			LineType.URBAN_MONORAIL -> R.drawable.monorail_black
			LineType.BUS_REGIO -> R.drawable.bus_black //TODO
			LineType.BUS_RAPID -> R.drawable.bus_black //TODO
			LineType.BUS_STOPPING -> R.drawable.bus_black //TODO
			LineType.BUS_LOCAL -> R.drawable.bus_black //TODO
			LineType.BUS_NIGHT -> R.drawable.bus_night_black
			LineType.BUS_POST -> R.drawable.bus_black //TODO
			LineType.BUS_SPECIAL_NEEDS -> R.drawable.bus_black //TODO
			LineType.BUS_MOBILITY -> R.drawable.bus_black //TODO
			LineType.BUS_MOBILITY_REGISTERED -> R.drawable.bus_black //TODO
			LineType.BUS_SIGHTSEEING -> R.drawable.bus_black //TODO
			LineType.BUS_SHUTTLE -> R.drawable.bus_black //TODO
			LineType.BUS_SCHOOL -> R.drawable.bus_school_black
			LineType.BUS_SCHOOL_PUBLIC_SERVICE -> R.drawable.bus_black //TODO
			LineType.BUS_RAIL_REPLACEMENT -> R.drawable.bus_black //TODO
			LineType.BUS_DEMAND -> R.drawable.bus_black //TODO
			LineType.BUS_ALL -> R.drawable.bus_black //TODO
			LineType.TRAM_CITY -> R.drawable.tram_black //TODO
			LineType.TRAM_LOCAL -> R.drawable.tram_black //TODO
			LineType.TRAM_REGIO -> R.drawable.tram_black //TODO
			LineType.TRAM_SIGHTSEEING -> R.drawable.tram_black //TODO
			LineType.TRAM_SHUTTLE -> R.drawable.tram_black //TODO
			LineType.TRAM_ALL -> R.drawable.tram_black //TODO
			LineType.WATER -> R.drawable.water_vehicle_black
			LineType.AERIAL_LIFT -> R.drawable.aerial_lift_black
			LineType.TELECABIN -> R.drawable.vehicle_black //TODO
			LineType.CABLE_CAR -> R.drawable.vehicle_black //TODO
			LineType.ELEVATOR -> R.drawable.vehicle_black //TODO
			LineType.CHAIR_LIFT -> R.drawable.chairlift_black
			LineType.DRAG_LIFT -> R.drawable.vehicle_black //TODO
			LineType.TELECABIN_SMALL -> R.drawable.vehicle_black //TODO
			LineType.TELECABIN_ALL -> R.drawable.vehicle_black //TODO
			LineType.TAXI -> R.drawable.taxi_black
			LineType.TAXI_COMMUNAL -> R.drawable.vehicle_black //TODO
			LineType.TAXI_WATER -> R.drawable.vehicle_black //TODO
			LineType.TAXI_RAIL -> R.drawable.vehicle_black //TODO
			LineType.TAXI_BIKE -> R.drawable.vehicle_black //TODO
			LineType.TAXI_LICENSED -> R.drawable.vehicle_black //TODO
			LineType.TAXI_PRIVATE_HIRE -> R.drawable.vehicle_black //TODO
			LineType.TAXI_ALL -> R.drawable.vehicle_black //TODO
			LineType.HORSE_CARIAGE -> R.drawable.vehicle_black //TODO
		}
		val icon = AppCompatResources.getDrawable(context, iconID)?.mutate()?.apply {
			setTint(textColour(colour))
		}?.toBitmap(dpToPixelI(19.2f / scale), dpToPixelI(19.2f / scale), Bitmap.Config.ARGB_8888)
		canvas.drawBitmap(
			icon!!, dpToPixel(2.4f / scale), dpToPixel(2.4f / scale), Paint()
		)
		return drawingBitmap
	}

	private fun getSquirclePath(
		left: Float, top: Float, radius: Int
	): Path {
		val radiusToPow = (radius * radius * radius).toDouble()
		val path = Path()
		path.moveTo(-radius.toFloat(), 0f)
		for (x in -radius..radius) path.lineTo(
			x.toFloat(), cbrt(radiusToPow - abs(x * x * x)).toFloat()
		)
		for (x in radius downTo -radius) path.lineTo(
			x.toFloat(), -cbrt(radiusToPow - abs(x * x * x)).toFloat()
		)
		path.close()
		val matrix = Matrix()
		matrix.postTranslate((left + radius), (top + radius))
		path.transform(matrix)
		return path
	}
}