// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.repo

import android.content.Context
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.api.AlertCauseV1
import xyz.apiote.bimba.czwek.api.AlertEffectV1
import xyz.apiote.bimba.czwek.api.AlertV1
import xyz.apiote.bimba.czwek.api.DepartureV1
import xyz.apiote.bimba.czwek.api.DepartureV2
import xyz.apiote.bimba.czwek.api.DepartureV3
import xyz.apiote.bimba.czwek.api.DepartureV4
import xyz.apiote.bimba.czwek.api.DepartureV5
import xyz.apiote.bimba.czwek.api.UnknownResourceVersionException
import xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.OTHER_CAUSE
import xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.UNKNOWN_CAUSE
import xyz.apiote.bimba.czwek.api.transitous.model.PickupDropoffType
import xyz.apiote.bimba.czwek.formatTimeAboutHM
import xyz.apiote.bimba.czwek.formatTimeAtHM
import xyz.apiote.bimba.czwek.formatTimeAtHMS
import xyz.apiote.bimba.czwek.formatTimeHM
import xyz.apiote.bimba.czwek.units.Second
import xyz.apiote.bimba.czwek.units.UnitSystem
import java.time.Instant
import java.time.ZoneId
import java.time.ZonedDateTime


class EventItem {
	private constructor(d: StopTime?, a: List<Alert>) {
		stopTime = d
		alert = a
	}

	constructor(d: StopTime) : this(d, emptyList())
	constructor(a: List<Alert>) : this(null, a)

	val stopTime: StopTime?
	val alert: List<Alert>
}

enum class AlertCause {
	UNKNOWN, OTHER, TECHNICAL_PROBLEM, STRIKE, DEMONSTRATION, ACCIDENT, HOLIDAY, WEATHER, MAINTENANCE,
	CONSTRUCTION, POLICE_ACTIVITY, MEDICAL_EMERGENCY;

	companion object {
		fun of(type: AlertCauseV1): AlertCause {
			return when (type) {
				AlertCauseV1.UNKNOWN -> valueOf("UNKNOWN")
				AlertCauseV1.OTHER -> valueOf("OTHER")
				AlertCauseV1.TECHNICAL_PROBLEM -> valueOf("TECHNICAL_PROBLEM")
				AlertCauseV1.STRIKE -> valueOf("STRIKE")
				AlertCauseV1.DEMONSTRATION -> valueOf("DEMONSTRATION")
				AlertCauseV1.ACCIDENT -> valueOf("ACCIDENT")
				AlertCauseV1.HOLIDAY -> valueOf("HOLIDAY")
				AlertCauseV1.WEATHER -> valueOf("WEATHER")
				AlertCauseV1.MAINTENANCE -> valueOf("MAINTENANCE")
				AlertCauseV1.CONSTRUCTION -> valueOf("CONSTRUCTION")
				AlertCauseV1.POLICE_ACTIVITY -> valueOf("POLICE_ACTIVITY")
				AlertCauseV1.MEDICAL_EMERGENCY -> valueOf("MEDICAL_EMERGENCY")
			}
		}

		fun ofTransitous(cause: xyz.apiote.bimba.czwek.api.transitous.model.AlertCause?): AlertCause {
			return when (cause) {
				UNKNOWN_CAUSE -> UNKNOWN
				OTHER_CAUSE -> OTHER
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.TECHNICAL_PROBLEM -> TECHNICAL_PROBLEM
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.STRIKE -> STRIKE
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.DEMONSTRATION -> DEMONSTRATION
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.ACCIDENT -> ACCIDENT
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.HOLIDAY -> HOLIDAY
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.WEATHER -> WEATHER
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.MAINTENANCE -> MAINTENANCE
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.CONSTRUCTION -> CONSTRUCTION
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.POLICE_ACTIVITY -> POLICE_ACTIVITY
				xyz.apiote.bimba.czwek.api.transitous.model.AlertCause.MEDICAL_EMERGENCY -> MEDICAL_EMERGENCY
				null -> UNKNOWN
			}
		}
	}
}

enum class AlertEffect {
	UNKNOWN, OTHER, NO_SERVICE, REDUCED_SERVICE, SIGNIFICANT_DELAYS, DETOUR, ADDITIONAL_SERVICE,
	MODIFIED_SERVICE, STOP_MOVED, NONE, ACCESSIBILITY_ISSUE;

	companion object {
		fun of(type: AlertEffectV1): AlertEffect {
			return when (type) {
				AlertEffectV1.UNKNOWN -> valueOf("UNKNOWN")
				AlertEffectV1.OTHER -> valueOf("OTHER")
				AlertEffectV1.NO_SERVICE -> valueOf("NO_SERVICE")
				AlertEffectV1.REDUCED_SERVICE -> valueOf("REDUCED_SERVICE")
				AlertEffectV1.SIGNIFICANT_DELAYS -> valueOf("SIGNIFICANT_DELAYS")
				AlertEffectV1.DETOUR -> valueOf("DETOUR")
				AlertEffectV1.ADDITIONAL_SERVICE -> valueOf("ADDITIONAL_SERVICE")
				AlertEffectV1.MODIFIED_SERVICE -> valueOf("MODIFIED_SERVICE")
				AlertEffectV1.STOP_MOVED -> valueOf("STOP_MOVED")
				AlertEffectV1.NONE -> valueOf("NONE")
				AlertEffectV1.ACCESSIBILITY_ISSUE -> valueOf("ACCESSIBILITY_ISSUE")
			}
		}

		fun ofTransitous(effect: xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect?): AlertEffect {
			return when (effect) {
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.NO_SERVICE -> NO_SERVICE
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.REDUCED_SERVICE -> REDUCED_SERVICE
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.SIGNIFICANT_DELAYS -> SIGNIFICANT_DELAYS
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.DETOUR -> DETOUR
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.ADDITIONAL_SERVICE -> ADDITIONAL_SERVICE
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.MODIFIED_SERVICE -> MODIFIED_SERVICE
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.OTHER_EFFECT -> OTHER
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.UNKNOWN_EFFECT -> UNKNOWN
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.STOP_MOVED -> STOP_MOVED
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.NO_EFFECT -> NONE
				xyz.apiote.bimba.czwek.api.transitous.model.AlertEffect.ACCESSIBILITY_ISSUE -> ACCESSIBILITY_ISSUE
				null -> UNKNOWN
			}
		}
	}
}

enum class AlertSource {
	SKNOW, AGENCY
}

data class Alert(
	val header: String,
	val description: String,
	val url: String,
	val cause: AlertCause,
	val effect: AlertEffect,
	val source: AlertSource
) {
	constructor(a: AlertV1) : this(
		a.header,
		a.Description,
		a.Url,
		AlertCause.of(a.Cause),
		AlertEffect.of(a.Effect),
		AlertSource.AGENCY
	)

	constructor(s: xyz.apiote.bimba.czwek.data.sknow.StopResponse, context: Context) : this(
		context.getString(R.string.sknow_alerts),
		"",
		"",
		AlertCause.UNKNOWN,
		AlertEffect.UNKNOWN,
		AlertSource.SKNOW
	)

	constructor(a: xyz.apiote.bimba.czwek.api.transitous.model.Alert) : this(
		a.headerText,
		a.descriptionText,
		a.url ?: "",
		AlertCause.ofTransitous(a.cause),
		AlertEffect.ofTransitous(a.effect),
		AlertSource.AGENCY
	)
}

data class StopEvents(
	val events: List<StopTime>,
	val stop: Stop,
	val alerts: List<Alert>
)

data class Event(
	val id: String,
	val arrivalTime: ZonedDateTime?,
	val scheduledArrival: ZonedDateTime?,
	val departureTime: ZonedDateTime?,
	val scheduledDeparture: ZonedDateTime?,
	val status: ULong,
	val isRealtime: Boolean,
	val vehicle: Vehicle,
	val boarding: UByte,
	val alerts: List<Alert>,
	val exact: Boolean,
	val terminusArrival: Boolean,  // TODO origin, middle, terminus; if origin -> only departure, if terminus -> only arrival
	val platform: String?,
	val platformScheduled: String?
) {

	companion object {
		fun boardingFromTransitous(pickup: PickupDropoffType?, dropoff: PickupDropoffType?): UByte {
			return when (pickup) {
				PickupDropoffType.NORMAL -> 0b0000_0001
				PickupDropoffType.NOT_ALLOWED -> 0
				null -> 0b0000_1111
			}.toUByte() or when (dropoff) {
				PickupDropoffType.NORMAL -> 0b0001_0000
				PickupDropoffType.NOT_ALLOWED -> 0
				null -> 0b1111_0000
			}.toUByte()
		}
	}

	constructor(d: DepartureV1) : this(
		d.ID,
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.status,
		d.isRealtime,
		Vehicle(d.vehicle),
		d.boarding,
		emptyList(),
		true,
		false,
		null,
		null
	)

	constructor(d: DepartureV2) : this(
		d.ID,
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.status,
		d.isRealtime,
		Vehicle(d.vehicle),
		d.boarding,
		emptyList(),
		true,
		false,
		null,
		null
	)

	constructor(d: DepartureV3) : this(
		d.ID,
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.status.ordinal.toULong(), // TODO VehicleStatus
		d.isRealtime,
		Vehicle(d.vehicle),
		d.boarding,
		emptyList(),
		true,
		false,
		null,
		null
	)

	constructor(d: DepartureV4) : this(
		d.ID,
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.status.ordinal.toULong(), // TODO VehicleStatus
		d.isRealtime,
		Vehicle(d.vehicle),
		d.boarding,
		d.alerts.map { Alert(it) },
		true,
		false,
		null,
		null
	)

	constructor(d: DepartureV5) : this(
		d.ID,
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.time.toZonedTime(),
		d.status.ordinal.toULong(), // TODO VehicleStatus
		d.isRealtime,
		Vehicle(d.vehicle),
		d.boarding,
		d.alerts.map { Alert(it) },
		d.exact,
		d.terminusArrival,
		null,
		null
	)

	fun timeZone() = (arrivalTime ?: departureTime)!!.zone.id

	fun filterTime() = (arrivalTime ?: departureTime)!!

	fun statusText(
		context: Context?,
		showAsTime: Boolean,
		at: ZonedDateTime? = null
	): Pair<String?, String?> {
		val now = at ?: Instant.now().atZone(ZoneId.systemDefault())
		return Pair(
			statusText(context, showAsTime, now, arrivalTime, R.string.departure_arrived),
			statusText(context, showAsTime, now, departureTime, R.string.departure_departed)
		)
	}

	private fun statusText(
		context: Context?,
		showAsTime: Boolean,
		now: ZonedDateTime,
		time: ZonedDateTime?,
		pastString: Int
	): String? {
		val r = status.toUInt()
		return time?.let {
			ZonedDateTime.of(
				time.year,
				time.monthValue,
				time.dayOfMonth,
				it.hour,
				it.minute,
				it.second,
				0,
				it.zone
			)
				.let { time ->
					if (showAsTime) {
						time.formatTimeHM()
					} else {
						when {
							r == 0u || (time.isBefore(now) && r < 3u) -> context?.let { context ->
								val us = UnitSystem.getSelected(context)
								us.toString(
									context,
									us.timeUnit(Second((time.toEpochSecond() - now.toEpochSecond()).toInt())),
									true
								)
							}

							r == 1u -> context?.getString(R.string.departure_momentarily) ?: "momentarily"
							r == 2u -> context?.getString(R.string.departure_now) ?: "now"
							r == 3u -> context?.getString(pastString) ?: "passed"
							else -> throw UnknownResourceVersionException("VehicleStatus/$r", 1u)
						}
					}
				}
		}
	}

	fun departureTimeString(context: Context): String? = timeString(context, departureTime)

	fun arrivalTimeString(context: Context): String? = timeString(context, arrivalTime)

	private fun timeString(context: Context, time: ZonedDateTime?): String? {
		return when {
			time == null -> null
			isRealtime -> time.formatTimeAtHMS(context)

			exact -> time.formatTimeAtHM(context)

			else -> time.formatTimeAboutHM(context)
		}

	}

	fun boardingText(context: Context): String {
		// todo [3.x] probably should take into account (on|off)-boarding only, on demand
		return when {
			boarding == (0b0000_0000).toUByte() -> context.getString(R.string.no_boarding)
			boarding == (0b1111_1111).toUByte() -> "" // unknown
			boarding.and(0b0011_0011u) == (0b0000_0001).toUByte() -> context.getString(R.string.on_boarding)
			boarding.and(0b0011_0011u) == (0b0001_0000).toUByte() -> context.getString(R.string.off_boarding)
			boarding.and(0b0011_0011u) == (0b0001_0001).toUByte() -> context.getString(R.string.boarding)
			else -> context.getString(R.string.on_demand)
		}
	}
}
