// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.network

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.api.Error
import xyz.apiote.bimba.czwek.data.exceptions.BimbaException
import xyz.apiote.bimba.czwek.data.exceptions.ClientException
import xyz.apiote.bimba.czwek.data.exceptions.TransientException
import java.security.KeyStore
import java.security.cert.CertificateFactory
import java.security.cert.X509Certificate
import javax.net.ssl.SSLContext
import javax.net.ssl.SSLSocketFactory
import javax.net.ssl.TrustManager
import javax.net.ssl.TrustManagerFactory
import javax.net.ssl.X509TrustManager

fun isNetworkAvailable(context: Context): Boolean {
	val connectivityManager =
		context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

	return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
		connectivityManager.activeNetwork?.let { network ->
			connectivityManager.getNetworkCapabilities(network)?.let { capabilities ->
				capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) || capabilities.hasTransport(
					NetworkCapabilities.TRANSPORT_CELLULAR
				) || capabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)
			}
		} == true
	} else {
		@Suppress("DEPRECATION")
		(connectivityManager.activeNetworkInfo?.isConnected == true)
	}
}

fun mapError(code: Int, message: String?): BimbaException {
	return when (code) {
		41 -> ClientException(
			message ?: "locality not supported", Error(code, R.string.error_41, R.drawable.error_locality)
		)

		44 -> ClientException(
			message ?: "", Error(code, R.string.error_44, R.drawable.error_departures)
		)

		400 -> ClientException(
			message ?: "wrong request",
			Error(code, R.string.error_400, R.drawable.error_app)
		)

		401 -> ClientException(
			message ?: "not authorised",
			Error(code, R.string.error_401, R.drawable.error_sec)
		)

		403 -> ClientException(
			message ?: "forbidden",
			Error(code, R.string.error_403, R.drawable.error_sec)
		)

		404 -> ClientException(
			message ?: "not found",
			Error(code, R.string.error_404, R.drawable.error_search)
		)

		406 -> ClientException(
			message ?: "not acceptable",
			Error(code, R.string.error_406, R.drawable.error_accept)
		)

		429 -> ClientException(
			message ?: "rate limited",
			Error(code, R.string.error_429, R.drawable.error_limit)
		)

		500 -> TransientException(
			message ?: "server error",
			Error(code, R.string.error_50x, R.drawable.error_server)
		)

		502 -> TransientException(
			message ?: "bad gateway",
			Error(code, R.string.error_50x, R.drawable.error_server)
		)

		503 -> TransientException(
			message ?: "service unavailable",
			Error(code, R.string.error_50x, R.drawable.error_server)
		)

		504 -> TransientException(
			message ?: "gateway timeout",
			Error(code, R.string.error_50x, R.drawable.error_server)
		)

		else -> ClientException(
			message ?: "unknown exception", Error(code, R.string.error_unknown, R.drawable.error_other)
		)
	}
}

fun getSslWithNewLetsEncryptRoot(context: Context): Pair<SSLSocketFactory, TrustManager> {
	val trustStore = KeyStore.getInstance(KeyStore.getDefaultType())
	trustStore.load(null, null)

	trustStore.setCertificateEntry(
		"let_s_encrypt_x1", CertificateFactory.getInstance("X509")
			.generateCertificate(context.resources.openRawResource(R.raw.isrgrootx1)) as X509Certificate?
	)
	trustStore.setCertificateEntry(
		"let_s_encrypt_x2", CertificateFactory.getInstance("X509")
			.generateCertificate(context.resources.openRawResource(R.raw.isrgrootx2)) as X509Certificate?
	)

	addDefaultRootCaCertificates(trustStore)
	return createSslSocketFactory(trustStore)
}

private fun createSslSocketFactory(trustStore: KeyStore?): Pair<SSLSocketFactory, TrustManager> {
	val trustManagers =
		TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm()).apply {
			init(trustStore)
		}.trustManagers

	return Pair(
		SSLContext.getInstance("SSL").apply {
			init(
				null,
				trustManagers,
				null
			)
		}.socketFactory,
		trustManagers[0]
	)
}

private fun addDefaultRootCaCertificates(trustStore: KeyStore) {
	for (trustManager in TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm())
		.apply {
			init(null as KeyStore?)
		}.trustManagers) {
		if (trustManager is X509TrustManager) {
			for (acceptedIssuer in trustManager.acceptedIssuers) {
				trustStore.setCertificateEntry(acceptedIssuer.subjectDN.name, acceptedIssuer)
			}
		}
	}
}
