// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.journeys

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Paint
import android.text.SpannableStringBuilder
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.appcompat.widget.TooltipCompat
import androidx.recyclerview.widget.RecyclerView
import androidx.recyclerview.widget.RecyclerView.ViewHolder
import com.google.android.material.card.MaterialCardView
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.formatDate
import xyz.apiote.bimba.czwek.formatTimeHM
import xyz.apiote.bimba.czwek.isDateSame
import xyz.apiote.bimba.czwek.isTimeSame
import xyz.apiote.bimba.czwek.repo.Journey
import xyz.apiote.bimba.czwek.repo.Stop.LineDecoration
import xyz.apiote.bimba.czwek.units.UnitSystem
import java.time.ZonedDateTime

class JourneysViewHolder(itemView: View) : ViewHolder(itemView) {
	val root: MaterialCardView = itemView.findViewById(R.id.journey)
	val startTime: TextView = itemView.findViewById(R.id.start_time)
	val lines: TextView = itemView.findViewById(R.id.lines)
	val endTime: TextView = itemView.findViewById(R.id.end_time)
	val legs: LinearLayout = itemView.findViewById(R.id.legs)
	val duration: TextView = itemView.findViewById(R.id.duration)

	companion object {
		fun bind(
			holder: JourneysViewHolder,
			onClickListener: (Journey, Int) -> Unit,
			journey: Journey,
			context: Context,
			inflater: LayoutInflater,
			isOpen: Boolean,
			position: Int
		) {
			holder.root.setOnClickListener {
				onClickListener(journey, position)
			}
			holder.startTime.text = journey.startTime.formatTimeHM()
			holder.endTime.text = journey.endTime.formatTimeHM()
			holder.lines.text = journey.legs.filter { it.start.vehicle.Line.name.isNotBlank() }
				.fold(SpannableStringBuilder("")) { acc, leg ->
					if (acc.isNotBlank()) {
						acc.append(" ")
					}
					val s = leg.start.vehicle.Line.decorate(
						SpannableStringBuilder(leg.start.vehicle.Line.shortName ?: leg.start.vehicle.Line.name),
						LineDecoration.COLOUR,
						null
					)
					acc.append(s)
					acc
				}
			UnitSystem.getSelected(context).let { us ->
				holder.duration.apply {
					text =
						us.toString(context, journey.duration, false)
					contentDescription =
						journey.duration.contentDescription(context, us.base, false)
				}
			}

			holder.legs.removeAllViews()
			val today = ZonedDateTime.now()
			journey.legs.forEach { leg ->
				@SuppressLint("InflateParams")
				val legView = if (leg.start.vehicle.Line.kind.isActive()) {
					inflater.inflate(R.layout.journey_leg_active, null, false)
				} else {
					inflater.inflate(R.layout.journey_leg_transit, null, false)
				}

				val legLineStart = legView.findViewById<ImageView>(R.id.leg_origin_image)
				val legLineMiddle = legView.findViewById<ImageView>(R.id.leg_line_image)
				val legLineMiddle2 = legView.findViewById<ImageView?>(R.id.leg_line_image2)
				val legLineEnd = legView.findViewById<ImageView>(R.id.leg_destination_image)
				legLineStart.setColorFilter(leg.start.vehicle.Line.colour.toInt())
				legLineMiddle.setColorFilter(leg.start.vehicle.Line.colour.toInt())
				legLineMiddle2?.setColorFilter(leg.start.vehicle.Line.colour.toInt())
				legLineEnd.setColorFilter(leg.start.vehicle.Line.colour.toInt())

				val legOrigin = legView.findViewById<TextView?>(R.id.leg_origin)
				legOrigin?.text = leg.origin.getName()

				val legOriginPlatform = legView.findViewById<TextView?>(R.id.leg_origin_platform)
				if (leg.start.platform != null) {
					legOriginPlatform?.text = context.getString(R.string.platform, leg.start.platform)
					legOriginPlatform?.visibility = View.VISIBLE
				} else {
					legOriginPlatform?.visibility = View.GONE
				}

				val legOriginPlatformScheduled =
					legView.findViewById<TextView?>(R.id.leg_origin_platform_scheduled)
				if (leg.start.platformScheduled != null && leg.start.platformScheduled != leg.start.platform) {
					legOriginPlatformScheduled?.paintFlags =
						legOriginPlatformScheduled.paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
					legOriginPlatformScheduled?.text = leg.start.platformScheduled
					legOriginPlatformScheduled?.visibility = View.VISIBLE
				} else {
					legOriginPlatformScheduled?.visibility = View.GONE
				}

				val legOriginTime = legView.findViewById<TextView>(R.id.leg_origin_time)
				legOriginTime.text = leg.start.departureTime!!.formatTimeHM()

				val legOriginDate = legView.findViewById<TextView>(R.id.leg_origin_time_date)
				legOriginDate.text = leg.start.departureTime.formatDate()
				if (leg.start.departureTime.isDateSame(today)) {
					legOriginDate.visibility = View.GONE
				} else {
					legOriginDate.visibility = View.VISIBLE
				}

				val legOriginTimeScheduled = legView.findViewById<TextView?>(R.id.leg_origin_time_scheduled)
				if (leg.start.scheduledDeparture != null && !leg.start.departureTime.isTimeSame(leg.start.scheduledDeparture)) {
					legOriginTimeScheduled?.paintFlags =
						legOriginTimeScheduled.paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
					legOriginTimeScheduled?.visibility = View.VISIBLE
					legOriginTimeScheduled?.text = leg.start.scheduledDeparture.formatTimeHM()
				} else {
					legOriginTimeScheduled?.visibility = View.GONE
				}

				val legOriginDateScheduled =
					legView.findViewById<TextView?>(R.id.leg_origin_time_date_scheduled)
				if (leg.start.scheduledDeparture != null && !leg.start.departureTime.isDateSame(leg.start.scheduledDeparture)) {
					legOriginDateScheduled?.paintFlags =
						legOriginDateScheduled.paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
					legOriginDateScheduled?.visibility = View.VISIBLE
					legOriginDate.visibility = View.VISIBLE
					legOriginDateScheduled?.text = leg.start.departureTime.formatDate()
				} else {
					legOriginDateScheduled?.visibility = View.GONE
				}

				val legModeImage = legView.findViewById<ImageView>(R.id.leg_mode_image)
				legModeImage.setImageDrawable(leg.start.vehicle.Line.icon(context))
				// TODO legModeImage.contentDescription = translate "leg mode: $mode"

				val serviceCancelled = legView.findViewById<TextView?>(R.id.service_cancelled)
				val legLine = legView.findViewById<TextView?>(R.id.leg_line)
				legLine?.text = leg.start.vehicle.Line.name
				if (leg.isCanceled) {
					legLine?.paintFlags = legLine.paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
					serviceCancelled?.visibility = View.VISIBLE
				} else {
					serviceCancelled?.visibility = View.GONE
				}

				val additionalServiceIcon = legView.findViewById<ImageView?>(R.id.additional_service_image)
				additionalServiceIcon?.visibility = if (leg.isScheduled) {
					View.GONE
				} else {
					View.VISIBLE
				}
				if (additionalServiceIcon != null) {
					TooltipCompat.setTooltipText(
						additionalServiceIcon,
						context.getString(R.string.additional_service)
					)
				}
				val realtimeIcon = legView.findViewById<ImageView?>(R.id.realtime_image)
				realtimeIcon?.visibility = if (leg.start.isRealtime) {
					View.VISIBLE
				} else {
					View.GONE
				}
				if (realtimeIcon != null) {
					TooltipCompat.setTooltipText(
						realtimeIcon,
						context.getString(R.string.realtime_content_description)
					)
				}

				val alertsButton = legView.findViewById<Button?>(R.id.alerts)
				// FIXME alerts should be on leg, not event
				Log.i("Alerts", "alerts: ${leg.start.alerts}")
				if (leg.start.alerts.isNotEmpty()) {
					alertsButton?.visibility = View.VISIBLE
					alertsButton?.setOnClickListener {
						MaterialAlertDialogBuilder(context)
							.setTitle(R.string.alerts)
							.setPositiveButton(R.string.ok) { _, _ -> }
							.setMessage(leg.start.alerts.joinToString(separator = "\n\n") { alert ->
								val description = alert.description.ifBlank {
									context.getString(R.string.no_description)
								}
								"${alert.header}:\n${description}"
							})
							.show()
					}
				} else {
					alertsButton?.visibility = View.GONE
				}

				val distance = legView.findViewById<TextView?>(R.id.leg_distance)
				leg.distance?.let { distance?.text = UnitSystem.getSelected(context).toString(context, it) }
				val stopNumber = legView.findViewById<TextView?>(R.id.leg_stop_number)
				stopNumber?.text = context.resources.getQuantityString(
					R.plurals.number_stops,
					leg.intermediateStops.size + 1,
					leg.intermediateStops.size + 1
				)
				val headsign = legView.findViewById<TextView?>(R.id.leg_headsign)
				if (leg.start.vehicle.Headsign.isNotBlank()) {
					headsign?.text = context.getString(R.string.line_headsign, leg.start.vehicle.Headsign)
					headsign?.visibility = View.VISIBLE
				} else {
					headsign?.visibility = View.GONE
				}

				legView.findViewById<LinearLayout?>(R.id.leg_intermediate_stops)?.let { intermediateStops ->
					leg.intermediateStops.forEach { stopTime ->
						@SuppressLint("InflateParams")
						val intermediateRow = inflater.inflate(R.layout.intermediate_row, null, false)
						val arrival = intermediateRow.findViewById<TextView>(R.id.arrival_time)
						stopTime.event.arrivalTime?.let {
							arrival.text = it.formatTimeHM()
							arrival.visibility = View.VISIBLE
						} ?: {
							arrival.visibility = View.GONE
						}
						val departure = intermediateRow.findViewById<TextView>(R.id.departure_time)
						if (stopTime.event.departureTime == null || stopTime.event.departureTime == stopTime.event.arrivalTime) {
							departure.visibility = View.GONE
						} else {
							departure.visibility = View.VISIBLE
							departure.text =
								stopTime.event.departureTime.formatTimeHM()
						}
						val stopName = intermediateRow.findViewById<TextView>(R.id.stop)
						stopName.text = stopTime.place.getName()

						val icon = intermediateRow.findViewById<ImageView>(R.id.stop_icon)
						icon.setColorFilter(leg.start.vehicle.Line.colour.toInt())
						with(stopTime.event.boarding) {
							when {
								this and 0b0001_0001.toUByte() == 0b0001_0001.toUByte() -> {
									icon.maxWidth = icon.width
									icon.visibility = View.VISIBLE
									stopName.alpha = 1f
								}

								this == 0.toUByte() -> {
									icon.maxWidth = icon.width
									icon.visibility = View.INVISIBLE
									stopName.alpha = 0.66f
								}

								// TODO yyyy…
								/*(this xor (this.toUInt().shl(4)).shr(4).toUByte()) == 1.toUByte() -> {
									icon.maxWidth = icon.width / 2
									icon.visibility = View.VISIBLE
									stopName.alpha = 1f
								}*/
							}
						}

						intermediateStops.addView(intermediateRow)
						intermediateStops.visibility = View.GONE
					}
					val expandButton = legView.findViewById<Button>(R.id.expand)
					val f = {
						if (!leg.start.vehicle.Line.kind.isActive()) {
							val isVisible = intermediateStops.visibility
							intermediateStops.visibility =
								if (isVisible == View.GONE) View.VISIBLE else View.GONE
							expandButton.rotation += 180
						}
					}
					if (leg.intermediateStops.isNotEmpty()) {
						expandButton.visibility = View.VISIBLE
						stopNumber?.setOnClickListener { f() }
						expandButton.setOnClickListener { f() }
					} else {
						expandButton.visibility = View.INVISIBLE
					}
				}

				val legDestination = legView.findViewById<TextView?>(R.id.leg_destination)
				legDestination?.text = leg.destination.getName()

				val legDestinationPlatform = legView.findViewById<TextView?>(R.id.leg_destination_platform)
				if (leg.end.platform != null) {
					legDestinationPlatform?.text = context.getString(R.string.platform, leg.end.platform)
					legDestinationPlatform?.visibility = View.VISIBLE
				} else {
					legDestinationPlatform?.visibility = View.GONE
				}

				val legDestinationPlatformScheduled =
					legView.findViewById<TextView?>(R.id.leg_destination_platform_scheduled)
				if (leg.end.platformScheduled != null && leg.end.platformScheduled != leg.end.platform) {
					legDestinationPlatformScheduled?.paintFlags =
						legDestinationPlatformScheduled.paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
					legDestinationPlatformScheduled?.text = leg.end.platformScheduled
					legDestinationPlatformScheduled?.visibility = View.VISIBLE
				} else {
					legDestinationPlatformScheduled?.visibility = View.GONE
				}

				val legDestinationTime = legView.findViewById<TextView>(R.id.leg_destination_time)
				legDestinationTime.text = leg.end.arrivalTime!!.formatTimeHM()

				val legDestinationDate = legView.findViewById<TextView>(R.id.leg_destination_time_date)
				legDestinationDate.text = leg.end.arrivalTime.formatDate()
				if (leg.end.arrivalTime.isDateSame(today)) {
					legDestinationDate.visibility = View.GONE
				} else {
					legDestinationDate.visibility = View.VISIBLE
				}

				val legDestinationTimeScheduled =
					legView.findViewById<TextView?>(R.id.leg_destination_time_scheduled)
				if (leg.end.scheduledArrival != null && !leg.end.arrivalTime.isTimeSame(leg.end.scheduledArrival)) {
					legDestinationTimeScheduled?.paintFlags =
						legDestinationTimeScheduled.paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
					legDestinationTimeScheduled?.visibility = View.VISIBLE
					legDestinationTimeScheduled?.text = leg.end.scheduledArrival.formatTimeHM()
				} else {
					legDestinationTimeScheduled?.visibility = View.GONE
				}

				val legDestinationDateScheduled =
					legView.findViewById<TextView?>(R.id.leg_destination_time_date_scheduled)
				if (leg.end.scheduledArrival != null && !leg.end.arrivalTime.isDateSame(leg.end.scheduledArrival)) {
					legDestinationDateScheduled?.paintFlags =
						legDestinationDateScheduled.paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
					legDestinationDateScheduled?.visibility = View.VISIBLE
					legDestinationDate.visibility = View.VISIBLE
					legDestinationDateScheduled?.text =
						leg.end.arrivalTime.formatDate()
				} else {
					legDestinationDateScheduled?.visibility = View.GONE
				}

				holder.legs.addView(legView)
			}
			holder.legs.visibility = if (isOpen) View.VISIBLE else View.GONE
		}
	}
}

class JourneysAdapter(
	private val inflater: LayoutInflater,
	private val context: Context,
	private var items: List<Journey>,
	private val onClickListener: ((Journey, Boolean) -> Unit),
) :
	RecyclerView.Adapter<JourneysViewHolder>() {
	var openCard: Int = -1

	val onClickListener2: ((Journey, Int) -> Unit) = { journey, position ->
		val previouslyOpen = openCard
		openCard = if (position == openCard) -1 else position
		notifyItemChanged(previouslyOpen)
		notifyItemChanged(position)
		onClickListener(journey, openCard == -1)
	}

	override fun onCreateViewHolder(
		parent: ViewGroup,
		viewType: Int
	): JourneysViewHolder {
		val rowView = inflater.inflate(R.layout.journey, parent, false)
		return JourneysViewHolder(rowView)
	}

	override fun onBindViewHolder(
		holder: JourneysViewHolder,
		position: Int
	) {
		JourneysViewHolder.bind(
			holder,
			onClickListener2,
			items[position],
			context,
			inflater,
			openCard == position,
			position
		)
	}

	override fun getItemCount(): Int = items.size
}