// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.departures

import android.content.Context
import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.TextView
import android.widget.Toast
import com.google.android.material.bottomsheet.BottomSheetDialogFragment
import com.google.android.material.button.MaterialButton
import com.google.android.material.textview.MaterialTextView
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.launch
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.data.sknow.InfrastructureDistortion
import xyz.apiote.bimba.czwek.data.sknow.InfrastructureDistortionID
import xyz.apiote.bimba.czwek.data.sknow.SknowRepository
import xyz.apiote.bimba.czwek.data.sknow.StopResponse
import xyz.apiote.bimba.czwek.data.sknow.StopUpdate
import xyz.apiote.bimba.czwek.formatDateTime
import xyz.apiote.bimba.czwek.repo.Stop
import java.time.Instant
import java.time.ZoneId
import java.time.ZonedDateTime
import kotlin.math.max

class SknowStopSheet(val stop: Stop) : BottomSheetDialogFragment() {
	private val stops = Channel<StopResponse>(Channel.UNLIMITED)
	private val updates = Channel<StopUpdate>(Channel.UNLIMITED)
	private var stopResponse: StopResponse? = null
	private var stopUpdate: StopUpdate = StopUpdate.empty()
	private var listener: ((StopResponse) -> Unit)? = null

	private var resetButton: MaterialButton? = null
	private var applyButton: MaterialButton? = null

	private var stopName: TextView? = null
	private var stopDescription: TextView? = null
	private var stopLevel: TextView? = null

	private var buttonWastebasketFull: MaterialButton? = null
	private var buttonWastebasketBroken: MaterialButton? = null
	private var lastUpdatedWastebasket: TextView? = null

	private var buttonShelterBroken: MaterialButton? = null
	private var lastUpdatedShelter: TextView? = null

	private var buttonBenchBroken: MaterialButton? = null
	private var lastUpdatedBench: TextView? = null

	private var buttonTimetableBroken: MaterialButton? = null
	private var buttonTimetableMissing: MaterialButton? = null
	private var lastUpdatedTimetable: TextView? = null

	private var buttonRtidBroken: MaterialButton? = null
	private var lastUpdatedRtid: TextView? = null

	private suspend fun startSknow(context: Context) {
		val repository = SknowRepository()
		if (!repository.isStopRunning(stop.stopPosition)) {
			SknowRepository().startStop(
				context,
				stop.stopPosition,
				updates,
				stops
			)
		}
	}

	private fun stopSknow() {
		SknowRepository().stopStop(stop.stopPosition)
	}

	fun setListener(f: (StopResponse) -> Unit) {
		listener = f
	}


	override fun onCreateView(
		inflater: LayoutInflater,
		container: ViewGroup?,
		savedInstanceState: Bundle?
	): View? {
		val content = inflater.inflate(R.layout.sheet_sknow_stop, container, false)
		content.apply {
			stopName = findViewById(R.id.stop_name)
			stopName?.text = stop.stopName

			stopDescription = findViewById<MaterialTextView>(R.id.stop_description)
			stopDescription?.text = stop.description

			stopLevel = findViewById<MaterialTextView>(R.id.stop_level)
			if (stop.level != null) {
				stopLevel?.text = context.getString(R.string.level, stop.level)
			} else {
				stopLevel?.text = ""
			}

			resetButton = findViewById(R.id.button_reset)
			resetButton?.setOnClickListener {
				Log.i("sknow", "reset: stop $stopResponse")
				applyStopToView()
			}
			applyButton = findViewById(R.id.button_save)
			applyButton?.isEnabled = false
			applyButton?.setOnClickListener {
				MainScope().launch {
					stopUpdate = stopUpdate.copy(SkidderID = "TODO", T = ZonedDateTime.now().toEpochSecond())
					updates.send(stopUpdate)
					stopUpdate = StopUpdate.empty()
					applyButton?.isEnabled = false
				}
			}

			buttonWastebasketFull = findViewById(R.id.wastebasket_full)
			lastUpdatedWastebasket = findViewById(R.id.wastebasket_last_update)
			buttonWastebasketFull?.setOnClickListener { v ->
				onInfraClick((v as MaterialButton).isChecked, InfrastructureDistortionID.BIN_FULL)
			}
			buttonWastebasketBroken = findViewById(R.id.wastebasket_broken)
			buttonWastebasketBroken?.setOnClickListener { v ->
				onInfraClick((v as MaterialButton).isChecked, InfrastructureDistortionID.BIN_BROKEN)
			}
			buttonShelterBroken = findViewById(R.id.shelter_broken)
			lastUpdatedShelter = findViewById(R.id.shelter_last_update)
			buttonShelterBroken?.setOnClickListener { v ->
				onInfraClick((v as MaterialButton).isChecked, InfrastructureDistortionID.SHELTER_BROKEN)
			}
			buttonBenchBroken = findViewById(R.id.bench_broken)
			lastUpdatedBench = findViewById(R.id.bench_last_update)
			buttonBenchBroken?.setOnClickListener { v ->
				onInfraClick((v as MaterialButton).isChecked, InfrastructureDistortionID.BENCH_BROKEN)
			}
			buttonTimetableBroken = findViewById(R.id.timetable_broken)
			lastUpdatedTimetable = findViewById(R.id.timetable_last_update)
			buttonTimetableBroken?.setOnClickListener { v ->
				onInfraClick((v as MaterialButton).isChecked, InfrastructureDistortionID.TIMETABLE_BROKEN)
			}
			buttonTimetableMissing = findViewById(R.id.timetable_missing)
			buttonTimetableMissing?.setOnClickListener { v ->
				onInfraClick((v as MaterialButton).isChecked, InfrastructureDistortionID.TIMETABLE_MISSING)
			}
			buttonRtidBroken = findViewById(R.id.rtid_broken)
			lastUpdatedRtid = findViewById(R.id.rtid_last_update)
			buttonRtidBroken?.setOnClickListener { v ->
				onInfraClick(
					(v as MaterialButton).isChecked,
					InfrastructureDistortionID.RTID_NOT_FUNCTIONING
				)
			}
		}
		return content
	}

	private fun onInfraClick(value: Boolean, kind: InfrastructureDistortionID) {
		applyButton?.isEnabled = true
		val infra = stopUpdate.Infrastructure.toMutableList()
		infra.apply {
			removeIf { it.Kind == kind }
			add(InfrastructureDistortion(value, kind))
			stopUpdate = stopUpdate.copy(Infrastructure = infra)
		}

	}

	override fun onResume() {
		super.onResume()
		start()
	}

	override fun onCreate(savedInstanceState: Bundle?) {
		super.onCreate(savedInstanceState)
		start()
	}

	private fun start() {
		context?.let {
			MainScope().launch {
				startSknow(it)
			}
			MainScope().launch {
				receiveUpdates(it)
			}
		}
		applyStopToView()
	}

	override fun onDestroy() {
		super.onDestroy()
		stopSknow()
	}

	private fun applyStopToView() {
		buttonWastebasketFull?.isChecked =
			stopResponse?.Infrastructure?.contains(InfrastructureDistortionID.BIN_FULL)
				?: false
		buttonWastebasketBroken?.isChecked =
			stopResponse?.Infrastructure?.contains(InfrastructureDistortionID.BIN_BROKEN)
				?: false
		max(
			stopResponse?.Infrastructure?.get(InfrastructureDistortionID.BIN_FULL)?.T ?: 0,
			stopResponse?.Infrastructure?.get(InfrastructureDistortionID.BIN_BROKEN)?.T ?: 0
		).takeIf { it > 0 }.let { seconds ->
			if (seconds != null) {
				lastUpdatedWastebasket?.text = getString(
					R.string.last_update,
					ZonedDateTime.ofInstant(Instant.ofEpochSecond(seconds), ZoneId.systemDefault())
						.formatDateTime()
				)
			} else {
				lastUpdatedWastebasket?.text = ""
			}
		}

		buttonShelterBroken?.isChecked =
			stopResponse?.Infrastructure?.contains(InfrastructureDistortionID.SHELTER_BROKEN)
				?: false
		stopResponse?.Infrastructure?.get(InfrastructureDistortionID.SHELTER_BROKEN)?.T.let { seconds ->
			if (seconds != null) {
				lastUpdatedShelter?.text = getString(
					R.string.last_update,
					ZonedDateTime.ofInstant(Instant.ofEpochSecond(seconds), ZoneId.systemDefault())
						.formatDateTime()
				)
			} else {
				lastUpdatedShelter?.text = ""
			}
		}

		buttonBenchBroken?.isChecked =
			stopResponse?.Infrastructure?.contains(InfrastructureDistortionID.BENCH_BROKEN)
				?: false
		stopResponse?.Infrastructure?.get(InfrastructureDistortionID.BENCH_BROKEN)?.T.let { seconds ->
			if (seconds != null) {
				lastUpdatedBench?.text = getString(
					R.string.last_update,
					ZonedDateTime.ofInstant(Instant.ofEpochSecond(seconds), ZoneId.systemDefault())
						.formatDateTime()
				)
			} else {
				lastUpdatedBench?.text = ""
			}
		}

		buttonTimetableBroken?.isChecked =
			stopResponse?.Infrastructure?.contains(InfrastructureDistortionID.TIMETABLE_BROKEN)
				?: false
		buttonTimetableMissing?.isChecked =
			stopResponse?.Infrastructure?.contains(InfrastructureDistortionID.TIMETABLE_MISSING)
				?: false
		max(
			stopResponse?.Infrastructure?.get(InfrastructureDistortionID.TIMETABLE_MISSING)?.T ?: 0,
			stopResponse?.Infrastructure?.get(InfrastructureDistortionID.TIMETABLE_BROKEN)?.T ?: 0
		).takeIf { it > 0 }.let { seconds ->
			if (seconds != null) {
				lastUpdatedTimetable?.text = getString(
					R.string.last_update,
					ZonedDateTime.ofInstant(Instant.ofEpochSecond(seconds), ZoneId.systemDefault())
						.formatDateTime()
				)
			} else {
				lastUpdatedTimetable?.text = ""
			}
		}

		buttonRtidBroken?.isChecked =
			stopResponse?.Infrastructure?.contains(InfrastructureDistortionID.RTID_NOT_FUNCTIONING)
				?: false
		stopResponse?.Infrastructure?.get(InfrastructureDistortionID.RTID_NOT_FUNCTIONING)?.T.let { seconds ->
			if (seconds != null) {
				lastUpdatedRtid?.text = getString(
					R.string.last_update,
					ZonedDateTime.ofInstant(Instant.ofEpochSecond(seconds), ZoneId.systemDefault())
						.formatDateTime()
				)
			} else {
				lastUpdatedRtid?.text = ""
			}
		}
	}

	@OptIn(DelicateCoroutinesApi::class)
	private suspend fun receiveUpdates(context: Context) {
		for (stop in stops) {
			Log.i("sknow", "received: $stop")
			if (stop.ErrorMessage != "") {
				if (stop.ErrorMessage == "success") {
					Toast.makeText(context, R.string.update_sent, Toast.LENGTH_LONG).show()
				} else {
					applyButton?.isEnabled = true
					Toast.makeText(
						context,
						getString(R.string.error_sending_update, stop.ErrorMessage), Toast.LENGTH_LONG
					).show()
				}
			} else {
				stopResponse = stop
				this.listener?.let { it(stop) }
				applyStopToView()
			}
			Log.i("sknow", "waiting for next message")
		}
	}
}