// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.data.sources

import android.content.Context
import android.util.Log
import io.ktor.client.HttpClient
import io.ktor.client.engine.cio.CIO
import io.ktor.client.plugins.websocket.WebSocketException
import io.ktor.client.plugins.websocket.WebSockets
import io.ktor.client.plugins.websocket.webSocket
import io.ktor.client.request.header
import io.ktor.websocket.Frame
import kotlinx.atomicfu.atomic
import kotlinx.coroutines.channels.ReceiveChannel
import kotlinx.coroutines.channels.SendChannel
import kotlinx.coroutines.selects.select
import kotlinx.serialization.decodeFromByteArray
import kotlinx.serialization.encodeToByteArray
import net.orandja.obor.codec.Cbor
import xyz.apiote.bimba.czwek.data.sknow.StopResponse
import xyz.apiote.bimba.czwek.data.sknow.StopResponseA
import xyz.apiote.bimba.czwek.data.sknow.StopUpdate
import xyz.apiote.bimba.czwek.network.getSslWithNewLetsEncryptRoot
import java.nio.channels.ClosedChannelException
import javax.net.ssl.X509TrustManager

class Sknow(
	val context: Context,
	val lat: Double,
	val lon: Double,
	val updates: ReceiveChannel<StopUpdate>,
	val stops: SendChannel<StopResponse>
) {
	val accept = "application/x.bimba.sknow.stops.alanais+cbor"

	// TODO disallow changing underlying value
	val running = atomic(false)

	private var websocketClient: HttpClient? = null

	init {
		if (websocketClient == null) {
			websocketClient = HttpClient(CIO) {
				engine {
					https {
						trustManager = getSslWithNewLetsEncryptRoot(context).second as X509TrustManager
					}
				}
				install(WebSockets)
			}
		}
	}

	fun stopStop() {
		websocketClient!!.close()
		running.value = false
	}

	suspend fun startStop() {
		// TODO catch errors
		// TODO add ping
		// TODO preflight head and select Content-Type and Accept
		try {
			websocketClient!!.webSocket(
				"wss://bimba.apiote.xyz/sknow/stops?lat=${lat}&lon=${lon}",
				request = {
					header("Content-Type", "application/x.bimba.sknow.stops.alanais+cbor")
					header("Accept", "application/x.bimba.sknow.stops.alanais+cbor")
//					User.load(context).authState?.accessToken?.let {
//						header("Authorization", "Bearer $it")
//					}
				}) {
				running.value = true
				while (true) {
					try {
						select<Unit> {
							incoming.onReceiveCatching { v ->
								val message = v.getOrNull()
								if (message == null) {
									Log.i("sknow", "incoming is closed")
									running.value = false
									throw ClosedChannelException()
								}
								Log.i("sknow", "new message in incoming: $message")
								(message as? Frame.Binary)?.data?.let { data ->
									val stopResponse = handleIncomingStop(data)
									Log.i("sknow", "sending through channel")
									stops.send(stopResponse)
									Log.i("sknow", "waiting for next message in incoming")
								}
							}

							updates.onReceiveCatching { v ->
								val update = v.getOrNull()
								if (update == null) {
									Log.i("sknow", "updates is closed")
									running.value = false
									throw ClosedChannelException()
								}
								Log.i("sknow", "new update: $update")
								val data = Cbor.encodeToByteArray(update)
								val frame = Frame.Binary(true, data)
								Log.i("sknow", "update sending")
								val s = outgoing.trySend(frame)
								Log.i("sknow", "update sent: $s")
							}
						}
					} catch (_: ClosedChannelException) {
						running.value = false
						break
					}
				}
			}
		} catch (e: WebSocketException) {
			running.value = false
			Log.e("sknow", "exception: $e")
		}
	}

	@OptIn(ExperimentalStdlibApi::class)
	private fun handleIncomingStop(data: ByteArray): StopResponse {
		val codec = Cbor {
			ingnoreUnknownKeys = true
		}

		Log.i("sknow", "incoming: ${data.toHexString()}")
		val stopResponse = when (accept) {
			"application/x.bimba.sknow.stops.alanais+cbor" -> StopResponse(
				codec.decodeFromByteArray<StopResponseA>(
					data
				)
			)

			else -> {
				Log.i("sknow", "unsupported tag in incoming")
				TODO("not supported")
			}
		}
		return stopResponse
	}
}
