// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.dashboard.ui.home

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.text.Editable
import android.text.TextWatcher
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.onStart
import kotlinx.coroutines.launch
import org.openapitools.client.infrastructure.ServerException
import xyz.apiote.bimba.czwek.data.exceptions.BimbaException
import xyz.apiote.bimba.czwek.data.settings.SettingsRepository
import xyz.apiote.bimba.czwek.data.traffic.TrafficRepository
import xyz.apiote.bimba.czwek.repo.Event
import xyz.apiote.bimba.czwek.repo.Favourite
import xyz.apiote.bimba.czwek.repo.OfflineRepository
import xyz.apiote.bimba.czwek.repo.OnlineRepository
import xyz.apiote.bimba.czwek.repo.Queryable
import xyz.apiote.bimba.czwek.repo.TrafficResponseException
import xyz.apiote.bimba.czwek.settings.feeds.FeedsSettings
import java.io.IOException
import java.net.UnknownHostException
import java.sql.SQLException
import java.util.Optional
import kotlin.jvm.optionals.getOrNull

class HomeViewModel : ViewModel() {
	private val mutableQueryables = MutableStateFlow(emptyList<Queryable>())
	val queryables: StateFlow<List<Queryable>> = mutableQueryables
	var feedsSettings: FeedsSettings? = null
	private val mutableFavourites = MutableLiveData<List<Favourite>>()
	val favourites: LiveData<List<Favourite>> = mutableFavourites
	private val mutableDepartures = MutableLiveData<Map<String, Optional<Event>>>()
	val departures: LiveData<Map<String, Optional<Event>>> = mutableDepartures

	fun getQueryables(query: String, context: Context) {
		viewModelScope.launch(Dispatchers.IO) {
			TrafficRepository()
				.queryQueryables(query, context)
				.onStart {
					mutableQueryables.value = emptyList<Queryable>()
				}
				.collect { flowResult ->
					when (flowResult) {
						is BimbaException -> {
							// XXX intentionally no error showing in suggestions
							Log.e("Suggestion", "queryQueryables threw $flowResult")
						}

						is Queryable -> {
							mutableQueryables.value = mutableQueryables.value.plus(flowResult)//.sortedBy {  }
						}
					}
				}
		}
	}

	fun getFavourites(context: Context) {
		viewModelScope.launch {
			try {
				getFeeds(context)
				val repository = OfflineRepository(context)
				val activeFeeds = (feedsSettings?.ids() ?: emptySet())
				mutableFavourites.value = repository.getFavourites(activeFeeds)
				repository.close()
			} catch (e: SQLException) {
				Log.w("FavouritesForFavourite", "$e")
			}
			getDeparturesOnly(context)
		}
	}

	fun getDepartures(context: Context) {
		viewModelScope.launch {
			getDeparturesOnly(context)
		}
	}

	private suspend fun getDeparturesOnly(context: Context) {
		coroutineScope {
			if (favourites.value == null)
				return@coroutineScope
			mutableDepartures.value = favourites.value!!.map { favourite ->
				async {
					try {
						val repository = OnlineRepository()
						val stopDepartures =
							repository.getDepartures(
								favourite.feedID,
								favourite.stopCode,
								null,
								context,
								12,  // XXX heuristics
								favourite.exact
							)
						stopDepartures?.let { sDs ->
							if (sDs.events.isEmpty()) {
								Pair(favourite.feedID + favourite.stopCode, Optional.empty())
							} else {
								Pair(
									favourite.feedID + favourite.stopCode,
									Optional.ofNullable(sDs.events.find { departure ->
										favourite.lines.isEmpty() or favourite.lines.contains(
											departure.event.vehicle.Line.name
										)
									})
								)
							}
						} ?: Pair(favourite.feedID + favourite.stopCode, Optional.empty())
					} catch (e: TrafficResponseException) {
						Log.w("DeparturesForFavourite", "$e")
						Pair(favourite.feedID + favourite.stopCode, Optional.empty())
					} catch (e: ServerException) {
						Log.w("DeparturesForFavourite", "Transitous returned ${e.statusCode}, ${e.message}")
						Pair(favourite.feedID + favourite.stopCode, Optional.empty())
					} catch (e: BimbaException) {
						Log.w("DeparturesForFavourite", "$e")
						// TODO do something better with exceptions in favourites on dashboard
						Pair(favourite.feedID + favourite.stopCode, Optional.empty())
					} catch (e: UnknownHostException) {
						Log.w("DeparturesForFavourite", "$e")
						// TODO do something better with exceptions in favourites on dashboard
						Pair(favourite.feedID + favourite.stopCode, Optional.empty())
					} catch (e: IOException) {
						Log.w("DeparturesForFavourite", "$e")
						// TODO do something better with exceptions in favourites on dashboard
						Pair(favourite.feedID + favourite.stopCode, Optional.empty())
					}
				}
			}.awaitAll().associate {
				Pair(it.first, Optional.ofNullable(it.second.getOrNull()?.event))
			}
		}
	}

	private fun getFeeds(context: Context) {
		feedsSettings = SettingsRepository().getAllFeeds(context)
	}

	fun saveFavourites(newFavourites: List<Favourite>, context: Context) {
		viewModelScope.launch {
			try {
				val repository = OfflineRepository(context)
				repository.saveFavourites(newFavourites.toSet())
				mutableFavourites.value = newFavourites
				repository.close()
			} catch (e: SQLException) {
				Log.w("FavouritesForFavourite", "$e")
			}
		}
	}


	inner class SearchBarWatcher(private val context: Context) : TextWatcher {
		private val handler = Handler(Looper.getMainLooper())
		private var workRunnable = Runnable {}

		override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
		}

		override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
		}

		override fun afterTextChanged(s: Editable?) {
			handler.removeCallbacks(workRunnable)
			workRunnable = Runnable {
				val text = s.toString()
				getQueryables(text, context)
			}
			handler.postDelayed(workRunnable, 750)
		}
	}
}
