// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.api

import android.content.Context
import androidx.preference.PreferenceManager
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.api.transitous.api.RoutingApi
import xyz.apiote.bimba.czwek.api.transitous.model.PedestrianProfile
import xyz.apiote.bimba.czwek.data.sources.Transitous
import xyz.apiote.bimba.czwek.data.traffic.Place
import xyz.apiote.bimba.czwek.network.isNetworkAvailable
import xyz.apiote.bimba.czwek.repo.Alert
import xyz.apiote.bimba.czwek.repo.Colour
import xyz.apiote.bimba.czwek.repo.CongestionLevel
import xyz.apiote.bimba.czwek.repo.Event
import xyz.apiote.bimba.czwek.repo.Journey
import xyz.apiote.bimba.czwek.repo.JourneyParams
import xyz.apiote.bimba.czwek.repo.Leg
import xyz.apiote.bimba.czwek.repo.LineStub
import xyz.apiote.bimba.czwek.repo.LineType
import xyz.apiote.bimba.czwek.repo.OccupancyStatus
import xyz.apiote.bimba.czwek.repo.Position
import xyz.apiote.bimba.czwek.repo.StopTime
import xyz.apiote.bimba.czwek.repo.TimeReference
import xyz.apiote.bimba.czwek.repo.TrafficResponseException
import xyz.apiote.bimba.czwek.repo.Vehicle
import xyz.apiote.bimba.czwek.units.Metre
import xyz.apiote.bimba.czwek.units.Mps
import xyz.apiote.bimba.czwek.units.Second
import java.time.ZoneId
import kotlin.math.pow

suspend fun getJourney(
	from: Place,
	to: Place,
	params: JourneyParams,
	context: Context
): List<Journey> {
	if (!isNetworkAvailable(context)) {
		throw TrafficResponseException(0, "", Error(0, R.string.error_offline, R.drawable.error_net))
	}

	return withContext(Dispatchers.IO) {
		val response = RoutingApi(
			basePath = PreferenceManager.getDefaultSharedPreferences(context).getString(
				Transitous.ADDRESS_KEY, Transitous.DEFAULT_ADDRESS
			) ?: Transitous.DEFAULT_ADDRESS,
			client = Transitous.client(context)
		).plan(
			from.getJourneyID(),
			to.getJourneyID(),
			maxTransfers = null,
			maxTravelTime = null,
			time = params.getDateTime(
				if (params.timeReference == TimeReference.DEPART_AFTER) {
					from.getTimezone()
				} else {
					to.getTimezone()
				}
			).toOffsetDateTime(),
			arriveBy = params.timeReference == TimeReference.ARRIVE_BY,
			requireBikeTransport = params.bicycle,
			pedestrianProfile = if (params.wheelchairAccessible) PedestrianProfile.WHEELCHAIR else PedestrianProfile.FOOT,
		)
		val journeys = response.itineraries.map {
			val legs: List<Leg> = it.legs.map { leg ->
				val fromZone = leg.from.tz?.let { ZoneId.of(it) } ?: ZoneId.systemDefault()
				val toZone = leg.to.tz?.let { ZoneId.of(it) } ?: ZoneId.systemDefault()
				Leg(
					Event(
						leg.tripId ?: "",
						null,
						null,
						leg.startTime.atZoneSameInstant(fromZone),
						leg.scheduledStartTime.atZoneSameInstant(fromZone),
						0u,
						leg.realTime,
						Vehicle(
							leg.tripId ?: "",
							Position(0.0, 0.0),
							0u,
							Mps(0),
							LineStub(
								leg.displayName ?: "",
								LineType.fromTransitous2(leg.mode, leg.routeType),
								Colour.fromHex(leg.routeColor),
								leg.routeShortName
							),
							leg.headsign ?: "",
							CongestionLevel.UNKNOWN,
							OccupancyStatus.UNKNOWN
						),
						boarding = Event.boardingFromTransitous(leg.from.pickupType, leg.from.dropoffType),
						alerts = leg.alerts?.map { alert ->
							Alert(alert)
						} ?: emptyList(),
						exact = true,
						terminusArrival = false,
						platform = leg.from.track,
						platformScheduled = leg.from.scheduledTrack
					),
					Event(
						leg.tripId ?: "",
						leg.endTime.atZoneSameInstant(toZone),
						leg.scheduledEndTime.atZoneSameInstant(toZone),
						null,
						null,
						0u,
						leg.realTime,
						Vehicle(
							leg.tripId ?: "",
							Position(0.0, 0.0),
							0u,
							Mps(0),
							LineStub(
								leg.displayName ?: "",
								LineType.fromTransitous2(leg.mode, leg.routeType),
								Colour.fromHex(leg.routeColor),
								leg.routeShortName
							),
							leg.headsign ?: "",
							CongestionLevel.UNKNOWN,
							OccupancyStatus.UNKNOWN
						),
						boarding = Event.boardingFromTransitous(leg.to.pickupType, leg.to.dropoffType),
						alerts = leg.alerts?.map { alert ->
							Alert(alert)
						} ?: emptyList(),
						exact = true,
						terminusArrival = false,
						platform = leg.to.track,
						platformScheduled = leg.to.scheduledTrack
					),
					Place.fromTransitousPlace(leg.from),
					Place.fromTransitousPlace(leg.to),
					leg.agencyName,
					leg.distance?.toDouble()?.let { distance -> Metre(distance) },
					Second(leg.duration),
					(leg.intermediateStops ?: emptyList()).map { stop -> StopTime(stop) },
					decode(leg.legGeometry.points, leg.legGeometry.precision),
					leg.scheduled,
					leg.cancelled == true
					/*it.rental,
					it.steps,*/
				)
			}
			Journey(
				it.startTime.atZoneSameInstant(it.legs[0].from.tz?.let { ZoneId.of(it) }
					?: ZoneId.systemDefault()),
				it.endTime.atZoneSameInstant(it.legs.last().to.tz?.let { ZoneId.of(it) }
					?: ZoneId.systemDefault()),
				legs, Second(it.duration)
			)
		}
		if (params.timeReference == TimeReference.ARRIVE_BY) {
			journeys.reversed()
		} else {
			journeys
		}
	}
}

/*
The following piece of code © Google, Apache License, Version 2.0
from https://github.com/googlemaps/android-maps-utils/blob/main/library/src/main/java/com/google/maps/android/PolyUtil.java
with changes to decode with dynamic precision
with fix to shift right with 0 and not sign bit
*/
fun decode(encodedPath: String, precision: Int = 6): MutableList<Position> {
	val exp = 10.0.pow(precision)
	val len = encodedPath.length

	val path: MutableList<Position> = ArrayList()
	var index = 0
	var lat = 0
	var lng = 0

	while (index < len) {
		var result = 1
		var shift = 0
		var b: Int
		do {
			b = encodedPath[index++].code - 63 - 1
			result += b shl shift
			shift += 5
		} while (b >= 0x1f)
		lat += if ((result and 1) != 0) (result ushr 1).inv() else (result ushr 1)

		result = 1
		shift = 0
		do {
			b = encodedPath[index++].code - 63 - 1
			result += b shl shift
			shift += 5
		} while (b >= 0x1f)
		lng += if ((result and 1) != 0) (result ushr 1).inv() else (result ushr 1)

		path.add(Position(lat / exp, lng / exp))
	}

	return path
}
