// SPDX-FileCopyrightText: Adam Evyčędo
//
// SPDX-License-Identifier: GPL-3.0-or-later

package xyz.apiote.bimba.czwek.api

import android.content.Context
import androidx.preference.PreferenceManager
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import xyz.apiote.bimba.czwek.R
import xyz.apiote.bimba.czwek.api.transitous.api.TimetableApi
import xyz.apiote.bimba.czwek.data.sources.Transitous
import xyz.apiote.bimba.czwek.network.isNetworkAvailable
import xyz.apiote.bimba.czwek.repo.Colour
import xyz.apiote.bimba.czwek.repo.CongestionLevel
import xyz.apiote.bimba.czwek.repo.Event
import xyz.apiote.bimba.czwek.repo.FeedInfo
import xyz.apiote.bimba.czwek.repo.LineStub
import xyz.apiote.bimba.czwek.repo.LineType
import xyz.apiote.bimba.czwek.repo.OccupancyStatus
import xyz.apiote.bimba.czwek.repo.Position
import xyz.apiote.bimba.czwek.repo.Stop
import xyz.apiote.bimba.czwek.repo.StopEvents
import xyz.apiote.bimba.czwek.repo.StopTime
import xyz.apiote.bimba.czwek.repo.TrafficResponseException
import xyz.apiote.bimba.czwek.repo.Vehicle
import xyz.apiote.bimba.czwek.units.Mps
import java.math.BigDecimal
import java.time.ZoneId
import java.time.ZonedDateTime
import java.time.temporal.ChronoUnit

suspend fun getTransitousDepartures(
	context: Context,
	stop: String,
	date: ZonedDateTime?,
	limit: Int?,
	exact: Boolean,
): StopEvents {
	if (!isNetworkAvailable(context)) {
		throw TrafficResponseException(0, "", Error(0, R.string.error_offline, R.drawable.error_net))
	}

	return withContext(Dispatchers.IO) {
		val datetime = date?.let {
			date.truncatedTo(ChronoUnit.DAYS).toOffsetDateTime()
		}
		val times =
			TimetableApi(
				basePath = PreferenceManager.getDefaultSharedPreferences(context).getString(
					Transitous.ADDRESS_KEY, Transitous.DEFAULT_ADDRESS
				) ?: Transitous.DEFAULT_ADDRESS,
				client = Transitous.client(context)
			).stoptimes(stop, limit ?: 12, datetime)
		var stopName = ""
		var latitude = BigDecimal(0)
		var longitude = BigDecimal(0)
		var zone = ZoneId.systemDefault()
		val stopTimes = times.stopTimes.filter {
			!exact || stop == it.place.stopId
		}.mapNotNull {
			if ((it.place.departure ?: it.place.arrival) == null) {
				null
			} else {
				latitude = it.place.lat
				longitude = it.place.lon
				stopName = it.place.name
				zone = it.place.tz?.let { ZoneId.of(it) } ?: ZoneId.systemDefault()
				val event = Event(
					it.tripId + it.source,
					it.place.arrival?.atZoneSameInstant(zone),
					null,
					it.place.departure?.atZoneSameInstant(zone),
					null,
					0u,
					it.realTime,
					Vehicle(
						it.tripId,
						Position(0.0, 0.0),
						0u,
						Mps(0),
						LineStub(
							it.routeShortName,
							LineType.fromTransitous2(it.mode, it.routeType),
							Colour.fromHex(it.routeColor)
						),
						it.headsign,
						CongestionLevel.UNKNOWN,
						OccupancyStatus.UNKNOWN
					),
					boarding = 0xffu,  // TODO
					alerts = emptyList(),
					exact = true,
					terminusArrival = it.place.departure == null,
					platform = it.place.track,
					platformScheduled = it.place.scheduledTrack
				)
				StopTime(Stop(it.place), event)
			}
		}
		StopEvents(
			stopTimes,
			Stop(
				stop,
				stopName,
				stopName,
				"",
				FeedInfo.ID_TRANSITOUS,
				Position(latitude.toDouble(), longitude.toDouble()),
				listOf(),
				null,
				null,
				zone,
			),
			listOf()
		)
	}
}
